from django.conf import settings
from wiki.api_v2.collections import Collection
from wiki.api_v2.di import di
from wiki.api_v2.exceptions import NotFound
from wiki.api_v2.public.pages.schemas import GroupWithMembersSchema, UserSchema, CheckOrgMembershipSchema
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users import dao
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.users_biz.consts import DIRECTORY_GROUP_TYPES

LIMIT_MEMBERS = 100


def serialize(grp):
    obj: GroupWithMembersSchema = GroupWithMembersSchema.serialize(grp)
    members = grp.get_all_members()[: (LIMIT_MEMBERS + 1)]
    has_next = len(members) > LIMIT_MEMBERS
    members = members[:LIMIT_MEMBERS]

    obj.members = Collection[UserSchema](
        results=[USER_DATA_REPOSITORY.orm_to_user_schema(q) for q in members], page_id=1, has_next=has_next
    )

    return obj


@di
def get_group_details_view(request, organization: BaseOrganization, group_id: str) -> GroupWithMembersSchema:
    """
    Возвращает информацию про департамент с `dir_id` (Только для B2B) или `staff_id` (Внутренний инстанс)
    """

    filter_params = {dao.get_id_attr_name(): group_id}

    if settings.IS_BUSINESS:
        filter_params['group_type'] = DIRECTORY_GROUP_TYPES.group

    obj = organization.get_groups().filter(**filter_params).first()

    if obj is None:
        raise NotFound()

    return serialize(obj)


@di
def get_department_details_view(request, organization: BaseOrganization, group_id: str) -> GroupWithMembersSchema:
    """
    Возвращает информацию про департамент с dir_id (Только для B2B)
    """

    if not settings.IS_BUSINESS:
        raise NotFound()

    filter_params = {dao.get_id_attr_name(): group_id, 'group_type': DIRECTORY_GROUP_TYPES.department}

    obj = organization.get_groups().filter(**filter_params).first()

    if obj is None:
        raise NotFound()

    return serialize(obj)


@di
def check_membership_view(request, organization: BaseOrganization, cloud_uid: str) -> CheckOrgMembershipSchema:
    """
    Возвращает информацию про департамент с dir_id (Только для B2B)
    """

    if not settings.IS_BUSINESS:
        raise NotFound()

    is_member = organization.get_users().filter(cloud_uid=cloud_uid).exists()
    return CheckOrgMembershipSchema(is_member=is_member)
