from datetime import datetime
from enum import Enum
from typing import Optional, List

from django.conf import settings
from ninja import Schema, Field
from pydantic import root_validator

from wiki.api_frontend.views.request_access import get_applicant_groups_and_departments
from wiki.api_v2.public.pages.schemas import UserSchema, GroupSchema
from wiki.api_v2.schemas import NotEmptyStringTrimmed
from wiki.users.user_data_repository import USER_DATA_REPOSITORY


class AccessRequestDecision(str, Enum):
    DENY = 'deny'
    ALLOW_APPLICANT = 'allow_applicant'
    ALLOW_GROUPS = 'allow_groups'


class AccessRequestDecisionEx(str, Enum):
    DENY = 'deny'
    ALLOW = 'allow'


class AccessRequestStatus(str, Enum):
    PENDING = 'pending'
    PROCESSED = 'processed'


class AccessRequestDecisionSchema(Schema):
    decision: AccessRequestDecisionEx
    processed_by: UserSchema
    comment: Optional[str]


class AccessRequestApplicantSchema(Schema):
    user: UserSchema
    departments: List[GroupSchema]
    services: List[GroupSchema]

    @classmethod
    def serialize(cls, user):
        departments, services = get_applicant_groups_and_departments(user)
        return AccessRequestApplicantSchema(
            user=USER_DATA_REPOSITORY.orm_to_user_schema(user),
            services=[GroupSchema.serialize(grp) for grp in services],
            departments=[GroupSchema.serialize(grp) for grp in departments],
        )


class CreateAccessRequestSchema(Schema):
    reason: NotEmptyStringTrimmed


class PageAccessRequestSchema(Schema):
    id: int
    status: AccessRequestStatus
    reason: str
    requested_at: datetime

    applicant: AccessRequestApplicantSchema
    resolution: Optional[AccessRequestDecisionSchema]

    @classmethod
    def serialize(cls, obj: 'AccessRequest'):
        return PageAccessRequestSchema(
            id=obj.pk,
            status=AccessRequestStatus.PENDING if obj.verdict is None else AccessRequestStatus.PROCESSED,
            reason=obj.reason,
            applicant=AccessRequestApplicantSchema.serialize(obj.applicant),
            requested_at=obj.created_at,
            resolution=None
            if obj.verdict is None
            else AccessRequestDecisionSchema(
                decision=AccessRequestDecisionEx.ALLOW if obj.verdict else AccessRequestDecisionEx.DENY,
                comment=obj.verdict_reason,
                processed_by=USER_DATA_REPOSITORY.orm_to_user_schema(obj.verdict_by),
            ),
        )


class ProcessAccessRequestSchema(Schema):
    decision: AccessRequestDecision
    reason: Optional[str]

    groups: Optional[List[int]] = Field(
        description='Для внешнего инстанса - dir_id, для внутреннего - staff_id', default_factory=list
    )
    departments: Optional[List[int]] = Field(description='Для внешнего инстанса - dir_id', default_factory=list)

    @root_validator
    def validator(cls, data):
        has_groups = bool(data.get('groups', None))
        has_depts = settings.IS_BUSINESS and bool(data.get('departments', None))

        if data['decision'] == AccessRequestDecision.ALLOW_GROUPS:
            assert has_groups or has_depts, 'Must provide at least one group'

        return data
