from itertools import chain

from django.shortcuts import get_object_or_404

from wiki.acl.check_access import has_access
from wiki.acl.consts import Action
from wiki.api_frontend.serializers.process_access_request import (
    set_resolution_deny,
    set_resolution_allow_applicant,
    set_resolution_allow_groups,
)
from wiki.api_frontend.utils.accessors import department_groups_by_ids, groups_by_ids
from wiki.api_v2.default_exception_handlers import upgrade_legacy_api_exceptions
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import Forbidden, IsStubPage
from wiki.api_v2.public.pages.access_requests.exceptions import AlreadyHasAccess, AlreadyProcessed
from wiki.api_v2.public.pages.access_requests.schemas import (
    ProcessAccessRequestSchema,
    PageAccessRequestSchema,
    CreateAccessRequestSchema,
    AccessRequestDecision,
)
from wiki.api_v2.public.pages.page.permissions import has_access_to_process_access_requests
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.notifications import signals
from wiki.pages.logic.etalon import create_etalon_page
from wiki.pages.models import AccessRequest
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils import timezone

CONTENT_TYPE_OCTET_STREAM = 'application/octet-stream'


@di
@legacy_org_ctx
def get_access_request_view(
    request, organization: BaseOrganization, idx: int, access_request_id: int
) -> PageAccessRequestSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    if not has_access_to_process_access_requests(request.user, organization, page):
        raise Forbidden()

    access_request = get_object_or_404(AccessRequest, pk=access_request_id, page=page)
    return PageAccessRequestSchema.serialize(access_request)


@di
@legacy_org_ctx
def create_access_request_view(
    request, organization: BaseOrganization, idx: int, data: CreateAccessRequestSchema
) -> PageAccessRequestSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    if has_access(request.user, organization, page, Action.READ):
        raise AlreadyHasAccess()

    access_request = AccessRequest.objects.filter(applicant=request.user, page=page, verdict_by=None).first()
    if access_request:
        access_request.created_at = timezone.now()
        access_request.reason = data.reason
    else:
        access_request = AccessRequest(applicant=request.user, page=page, reason=data.reason)

    access_request.save()
    signals.request_access_created(access_request)

    return PageAccessRequestSchema.serialize(access_request)


@di
@legacy_org_ctx
@upgrade_legacy_api_exceptions
def process_access_request_view(
    request, organization: BaseOrganization, idx: int, access_request_id: int, data: ProcessAccessRequestSchema
) -> PageAccessRequestSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    if not has_access_to_process_access_requests(request.user, organization, page):
        raise Forbidden()

    access_request: AccessRequest = get_object_or_404(AccessRequest, pk=access_request_id, page=page)

    if access_request.verdict is not None:
        raise AlreadyProcessed()

    if has_access(access_request.applicant, organization, page, Action.READ):
        raise AlreadyHasAccess()

    if data.decision == AccessRequestDecision.DENY:
        set_resolution_deny(access_request, data.reason, request.user)
    elif data.decision == AccessRequestDecision.ALLOW_APPLICANT:
        set_resolution_allow_applicant(access_request, request.user)
    elif data.decision == AccessRequestDecision.ALLOW_GROUPS:
        set_resolution_allow_groups(
            access_request,
            all_groups=chain(groups_by_ids(data.groups), department_groups_by_ids(data.departments)),
            user=request.user,
        )

    return PageAccessRequestSchema.serialize(access_request)
