import mimetypes

from datetime import datetime

from ninja import Schema
from pydantic import UUID4

from wiki.api_core.utils import is_docviewer_previewable
from wiki.api_v2.public.pages.schemas import UserSchema
from wiki.files.models import File
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.utils.docviewer import docviewer_preview_link


class AttachmentSchema(Schema):
    id: int
    name: str
    download_url: str
    docviewer_url: str | None
    size: str
    description: str
    user: UserSchema | None
    created_at: datetime
    mimetype: str
    has_preview: bool

    @classmethod
    def serialize(cls, file: File, page: Page, organization: BaseOrganization) -> 'AttachmentSchema':
        docviewer_url = (
            docviewer_preview_link(file, organization.external_id()) if is_docviewer_previewable(file) else None
        )

        return AttachmentSchema(
            id=file.id,
            name=file.name,
            download_url=file.full_path(page.url, file.url),
            docviewer_url=docviewer_url,
            size='%.2f' % (file.size / 1024.0 / 1024),  # in MB
            description=file.description,
            user=USER_DATA_REPOSITORY.orm_to_user_schema(file.user),
            created_at=file.created_at,
            mimetype=mimetypes.guess_type(file.name)[0] or 'application/octet-stream',
            has_preview=file.has_preview,
        )


class AttachFileSchema(Schema):
    upload_sessions: list[UUID4]


class AttachFileResponse(Schema):
    results: list[AttachmentSchema]
