import re
from functools import partial
from typing import Optional

from datetime import timedelta
from django.conf import settings
from django.http import HttpResponse
from django.shortcuts import get_object_or_404
from django_replicated.utils import routers
from ninja import Query
from pydantic import constr
from django.db import transaction

from wiki.acl.check_access import assert_has_access, has_access
from wiki.acl.consts import Action
from wiki.api_frontend.views.files import prepare_response
from wiki.api_v2.collections import Collection, PaginationQuery, CollectionFactory, OrderDirection
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import Forbidden, IsStubPage
from wiki.api_v2.public.pages.attachments.schemas import AttachFileResponse, AttachFileSchema, AttachmentSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.api_v2.schemas import DELETED
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.files import logic as files_logic
from wiki.files.api import file_download_response
from wiki.files.consts import ThumbnailFieldSchema, ThumbnailModel, ThumbnailStatus
from wiki.files.models import File
from wiki.files.tasks import CreateThumbnailTask
from wiki.pages import dao as pages_dao
from wiki.pages.access import is_admin
from wiki.pages.logic.etalon import create_etalon_page, get_etalon_page
from wiki.pages.logic.files import attach_file_from_upload_session
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.supertag import normalize_supertag
from wiki.utils.timezone import now
from wiki.uploads.models import UploadSession


base64_encoded_pixel = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mNkYAAAAAYAAjCB0C8AAAAASUVORK5CYII='


def assert_can_delete(user, page: Page, file: File):
    if user in page.get_authors() or user.id == file.user_id or is_admin(user):
        return
    raise Forbidden


ordering_field = {
    'name': ['name'],
    'size': ['size'],
    'created_at': ['created_at'],
}


CollectionAttachmentSchema = (
    CollectionFactory(name='attachment')
    .with_ordering(ordering_field)
    .default_ordering(['created_at'], direction=OrderDirection.DESC)
)


@di
@legacy_org_ctx
def attachments_view(
    request,
    organization: BaseOrganization,
    idx: int,
    ordering: CollectionAttachmentSchema.ordering = Query(...),
    pagination: PaginationQuery = Query(...),
) -> Collection[AttachmentSchema]:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    qs = pages_dao.get_files(page)

    return CollectionAttachmentSchema.ordered_build(
        qs=qs,
        ordering=ordering,
        serializer=partial(AttachmentSchema.serialize, organization=organization, page=page),
        pagination=pagination,
    )


@di
@transaction.atomic
def attach_file_view(request, organization: BaseOrganization, idx: int, data: AttachFileSchema) -> AttachFileResponse:
    """
    Добавление файла из upload_session на страницу
    """
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.WRITE)
    attachments = []
    for session_id in data.upload_sessions:
        upload_session: UploadSession = get_object_or_404(
            UploadSession.objects.select_for_update(),
            session_id=session_id,
            user=request.user,
            org=organization.as_django_model(),
        )

        attachment = attach_file_from_upload_session(page, request.user, upload_session)
        attachments.append(attachment)

    return AttachFileResponse(
        results=[AttachmentSchema.serialize(attachment, page, organization) for attachment in attachments]
    )


@di
@legacy_org_ctx
def delete_attach_view(request, organization: BaseOrganization, idx: int, file_id: int):
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    with transaction.atomic():
        attached_file = get_object_or_404(File.active.select_for_update(), pk=file_id, page=page)
        assert_can_delete(request.user, page, attached_file)
        files_logic.delete(attached_file, user=request.user, page=page)

    # Cloudsearch part:
    CLOUD_SEARCH_CLIENT.on_model_delete(attached_file)
    return DELETED


@di
def preview_by_file_id_view(request, organization: BaseOrganization, idx: int, file_id: int):
    """
    Получение превью для файлов изображений. В случае, если
    превью недоступно, например, файл не является изображением, будет
    возращен png файл размером 1 пиксель.
    """
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    file_obj = get_object_or_404(File.active, pk=file_id, page=page)

    thumbnail_data = file_obj.thumbnail_data
    if not File.is_image(file_obj.name) or thumbnail_data.status == ThumbnailStatus.NOT_AVAILABLE:
        return HttpResponse(base64_encoded_pixel, content_type='image/png')

    if thumbnail_data.status == ThumbnailStatus.CREATED:
        thumbnail = ThumbnailModel(
            name=f'{file_obj.id}.png',
            modified_at=file_obj.modified_at,
            s3_storage_key=thumbnail_data.s3_key,
        )
        return file_download_response(thumbnail, force_download=False)

    NOW = now()

    reschedule = False
    if thumbnail_data.status == ThumbnailStatus.SCHEDULED and (NOW - thumbnail_data.changed_at > timedelta(minutes=5)):
        reschedule = True

    if not thumbnail_data.status == ThumbnailStatus.SCHEDULED or reschedule:
        # Файл превью еще не был сгенерен или после
        # запуска предыдущей задачи прошло более пяти минут.
        try:
            routers.use_state('master')
            file_obj.thumbnail = ThumbnailFieldSchema(
                status=ThumbnailStatus.SCHEDULED,
                changed_at=NOW,
            ).serialize()
            file_obj.save()
            CreateThumbnailTask().delay(file_id=file_obj.id)
        finally:
            routers.revert()

    return HttpResponse(base64_encoded_pixel, content_type='image/png')


@di
def download_by_file_id_view(request, organization: BaseOrganization, idx: int, file_id: int):
    """
    Скачать файл по перманентному ID
    """
    maybe_file = File.objects.filter(page__org=organization.as_django_model(), page__id=idx, id=file_id).first()

    return _handle_download(maybe_file, 'tmp.png', request.user, organization, force_download=False)


legacy_file_url_regex = rf'^{settings.TAG_REGEX}/.files/(?P<filename>.+)$'


@di
def download_by_filename_slug_pair_view(
    request,
    organization: BaseOrganization,
    url: constr(regex=legacy_file_url_regex, min_length=1),
    download: bool = False,
):
    """
    Скачать файл по-старинке: по фронтовому url

    ручка --слишком-- умная - если страница со slug съехала (вместе с файлами)
    попробуем пройти по редиректам и другим местам

    download - хедеры обеспечат загрузку файла вместо отображения в браузере вне зависимости от MIME-типа
    """

    match = re.match(legacy_file_url_regex, url)
    groupdict = match.groupdict()

    slug = normalize_supertag(groupdict['tag'])
    filename = groupdict['filename']

    log_slug(request, slug=slug)

    try:
        page = organization.get_active_pages().get(supertag=slug)
        maybe_file = files_logic.file_for_page_by_filename(filename, page)
    except Page.DoesNotExist:
        maybe_file = files_logic.file_for_slug_by_filename(filename, organization, slug)

    return _handle_download(maybe_file, filename, request.user, organization, force_download=download)


def _handle_download(file: Optional[File], filename: str, user, organization, force_download: bool = False):
    if file is None:
        return prepare_response(404, File.is_image(filename))

    # Файл может быть найден на какой-то странице редиректа,
    # поэтому проверяем доступ к странице владеющей файлом
    # (а не той к которой было обращение)

    if not has_access(user, organization, file.page, Action.READ):
        return prepare_response(403, True)

    if file.storage_url is None:
        return prepare_response(404, File.is_image(file.name))  # битый недо-загруженный файл.

    return file_download_response(file, force_download)
