from ninja import Query
from wiki.api_v2.collections import Collection, PaginationQuery, CollectionFactory

from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action

from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.schemas import PageSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.pages.logic.etalon import get_etalon_page
from wiki.pages.models import PageLink
from wiki.sync.connect.base_organization import BaseOrganization


def serialize(x) -> PageSchema:
    return PageSchema(id=x[0], slug=x[1])


@di
@legacy_org_ctx
def backlinks_view(
    request, organization: BaseOrganization, idx: int, pagination: PaginationQuery = Query(...)
) -> Collection[PageSchema]:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    linked_pages = PageLink.objects.filter(
        to_page=page, from_page__status=1, from_page__org=organization.as_django_model()
    ).values_list('from_page__id', 'from_page__supertag', 'pk', named=True)

    return CollectionFactory.build(qs=linked_pages, serializer=serialize, pagination=pagination)
