from django.shortcuts import get_object_or_404
from ninja import Query
from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.collections import Collection, PaginationQuery, CollectionFactory, OrderDirection
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import Forbidden, IsStubPage
from wiki.api_v2.public.pages.comments.exceptions import CommentingDisabled, ParentCommentNotFound
from wiki.api_v2.public.pages.schemas import CommentSchema, CreateCommentSchema, EditCommentSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.api_v2.schemas import DELETED
from wiki.pages.access import is_admin
from wiki.pages.logic import comment as comment_logic
from wiki.pages.logic.comment import serialize_comment
from wiki.pages.logic.etalon import get_etalon_page, create_etalon_page
from wiki.pages.models import Comment, Page
from wiki.pages.models.consts import COMMENTS_STATUS
from wiki.sync.connect.base_organization import BaseOrganization


@di
def comments_view(
    request, organization: BaseOrganization, idx: int, pagination: PaginationQuery = Query(...)
) -> Collection[CommentSchema]:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    return (
        CollectionFactory()
        .default_ordering(['created_at'], direction=OrderDirection.DESC)
        .ordered_build(
            qs=page.comment_set.filter(status=True),
            serializer=serialize_comment,
            pagination=pagination,
        )
    )


@di
def get_comment_details_view(request, organization: BaseOrganization, idx: int, comment_id: int) -> CommentSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)
    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    comment = get_object_or_404(page.comment_set.all(), pk=comment_id, status=True)

    return serialize_comment(comment)


def assert_can_edit(user, comment: Comment):
    if is_admin(user) or user == comment.user:
        return
    raise Forbidden


def assert_can_delete(user, page: Page, comment: Comment):
    if is_admin(user) or user == comment.user or user in page.get_authors():
        return
    raise Forbidden


@di
def create_comment_view(request, organization: BaseOrganization, idx: int, data: CreateCommentSchema) -> CommentSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    if page.comments_status != COMMENTS_STATUS.enabled:
        raise CommentingDisabled

    parent_comment = None
    if data.parent_id:
        try:
            # поймает случаи:
            # если комментарий из чужой ветки или не существует или из другой организации
            parent_comment = page.comment_set.get(pk=data.parent_id)
        except Comment.DoesNotExist:
            raise ParentCommentNotFound

    comment = comment_logic.add_comment(
        user=request.user, page=page, body=data.body, parent_id=parent_comment and parent_comment.id
    )

    return serialize_comment(comment)


@di
@legacy_org_ctx
def edit_comment_view(request, organization: BaseOrganization, idx: int, comment_id: int, data: EditCommentSchema):
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)
    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    comment = get_object_or_404(page.comment_set, pk=comment_id, status=True)
    assert_can_delete(request.user, page, comment)

    comment_logic.edit_comment(comment=comment, user=request.user, body=data.body)

    return serialize_comment(comment)


@di
@legacy_org_ctx
def delete_comment_view(request, organization: BaseOrganization, idx: int, comment_id: int):
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    comment = get_object_or_404(page.comment_set, pk=comment_id, status=True)
    assert_can_delete(request.user, page, comment)

    comment_logic.delete_comment(
        comment=comment,
        user=request.user,
    )

    return DELETED
