from django.db.models import Q
from ninja import Query
from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_frontend.views.flat_list_pages import filter_by_access
from wiki.api_v2.collections import Collection, CollectionFactory
from wiki.api_v2.collections import MediumPaginationQuery
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.schemas import PageSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.pages.logic.etalon import get_etalon_page
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization


def serialize(page: Page) -> PageSchema:
    return PageSchema(id=page.id, slug=page.supertag)


@di
@legacy_org_ctx
def descendants_view(
    request,
    organization: BaseOrganization,
    idx: int,
    include_self: bool = False,
    pagination: MediumPaginationQuery = Query(...),
) -> Collection[PageSchema]:
    """
    Возвращает всех потомков (не только первого уровня, а вообще всех) данной страницы, доступных текущему пользователю.
    Ответ может содержать меньше результатов чем page_size, в зависимости от того ко скольким страницам есть доступ

    include_self - взять себя
    """
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    q = Q(supertag__startswith=page.supertag + '/')
    if include_self:
        q = q | Q(supertag=page.supertag)

    qs = organization.get_active_pages().filter(q)

    return CollectionFactory.build(
        qs=qs,
        serializer=serialize,
        pagination=pagination,
        filter=lambda x: list(filter_by_access(x, user=request.user)),
    )
