from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.consts import Location
from wiki.api_v2.public.pages.exceptions import (
    AnchorNotFound,
    SectionNotFound,
    UnknownLocation,
)
from wiki.api_v2.public.pages.page.details_serializer import PageCompositeSerializer
from wiki.api_v2.public.pages.page.schemas import PageAppendContentSchema
from wiki.api_v2.public.pages.page.set_content import update_page_content
from wiki.api_v2.public.pages.utils import make_redlock_tag
from wiki.api_v2.public.pages.schemas import PageFullDetailsSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.pages.logic.etalon import create_etalon_page
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils import timezone
from wiki.utils.django_redlock.redlock import RedisLock
from wiki.utils.wfaas.ast import ActionName
from wiki.utils.wfaas.client import get_wfaas_client


@di
@legacy_org_ctx
def append_content_view(
    request,
    organization: BaseOrganization,
    idx: int,
    data: PageAppendContentSchema,
    fields: str = None,
    is_silent: bool = False,
) -> PageFullDetailsSchema:
    with RedisLock(make_redlock_tag('append_content', organization, str(idx)), blocking_timeout=3):
        try:
            page = get_page_or_404(organization, pk=idx)
        except IsStubPage:
            page = create_etalon_page(organization, idx=idx)

        log_slug(request, slug=page.slug)
        assert_has_access(request.user, organization, page, Action.WRITE)

        if data.body:
            new_body = append_to_body(page, data.content, data.body.location)
        elif data.section:
            new_body = append_to_section(page, data.content, data.section.id, data.section.location)
        elif data.anchor:
            new_body = append_to_anchor(page, data.content, data.anchor.name)

        update_page_content(request.user, organization, page, new_body, is_silent)
        fields = PageCompositeSerializer.extract_fields(fields)
        page.modified_at = timezone.now()
        page.save()

    result: PageFullDetailsSchema = PageCompositeSerializer.serialize_page(page, fields, request, organization)
    return result


def append_to_body(
    page: Page,
    content: str,
    location: Location,
) -> str:
    match location:
        case Location.TOP:
            return insert_content(page.body, content, 0)
        case Location.BOTTOM:
            return insert_content(page.body, content, len(page.body))
        case _:
            raise UnknownLocation


def append_to_section(
    page: Page,
    content: str,
    section_id: int,
    location: Location,
) -> str:
    ast = get_wfaas_client().raw_to_ast(page.body)
    section = ast.get_section(section_id)
    if section.pos_start is None:
        raise SectionNotFound

    match location:
        case Location.TOP:
            return insert_content(page.body, content, section.pos_heading_end)
        case Location.BOTTOM:
            return insert_content(page.body, content, section.pos_end)
        case _:
            raise UnknownLocation


def append_to_anchor(
    page: Page,
    content: str,
    anchor_name: str,
) -> str:
    wfaas_settings = {
        'page_path': page.supertag,
        'remark': {
            'woofmd': {
                'actions': [{'name': ActionName.ANCHOR}, {'name': ActionName.ANCHOR_SHORT}],
            },
        },
    }
    ast = get_wfaas_client().raw_to_ast(page.body, wfaas_settings=wfaas_settings)
    anchor = ast.get_anchor(anchor_name)
    if anchor.pos_start is None:
        raise AnchorNotFound

    return insert_content(page.body, content, anchor.pos_end, delimiter_left='')


def insert_content(
    body: str,
    content: str,
    pos: int,
    delimiter_left: str = '\n',
    delimiter_right: str = '\n',
) -> str:
    left = body[:pos]
    right = body[pos:]
    if not left or left.endswith(delimiter_left) or content.startswith(delimiter_left):
        delimiter_left = ''
    if not right or right.startswith(delimiter_right) or content.endswith(delimiter_right):
        delimiter_right = ''
    return ''.join(
        (
            left,
            delimiter_left,
            content,
            delimiter_right,
            right,
        ),
    )
