from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.page_identity import PageIdentity
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.async_operations.consts import OperationCreatedSchema, OperationType
from wiki.async_operations.operation_executors.clone_page.schemas import PageCloneRequest, PageClone
from wiki.async_operations.views import schedule_execution_view
from wiki.pages.logic.etalon import create_etalon_page
from wiki.sync.connect.base_organization import BaseOrganization


@di
@legacy_org_ctx
def clone_page_view(
    request,
    organization: BaseOrganization,
    idx: int,
    data: PageCloneRequest,
) -> OperationCreatedSchema:
    """
    Клонирование страницы на новое место.

    Отложенная операция. Если все проверки пройдены, вернет информацию о том, как проверять статус

    Коды ошибок валидации (error_code)
    - `IS_CLOUD_PAGE`: нельзя клонировать облачные страницы
    - `SLUG_OCCUPIED`: после клонирования страница пересекается с уже существующей
    - `SLUG_RESERVED`: нельзя клонировать в зарезервированные страницы
    - `FORBIDDEN`: Нет доступа к исходной странице или к кластеру, в который клонируется страница
    - `QUOTA_EXCEEDED`: Достигнут лимит на создание страниц в текущей организации
    - `CLUSTER_BLOCKED`: Кластер временно заблокирован для переноса.
    """

    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    copy_data = PageClone(page=PageIdentity.serialize(page), data=data)
    return schedule_execution_view(request, organization, data=copy_data, op_type=OperationType.CLONE_PAGE)
