from wiki.api_v2.default_exception_handlers import upgrade_legacy_api_exceptions
from wiki.api_v2.exceptions import BadRequest
from wiki.api_v2.public.pages.page.consts import OperationResult
from wiki.api_v2.public.pages.page.create.create_regular_page import create_page
from wiki.api_v2.public.pages.page.schemas import CreatePageSchema, Ms365UploadSessionSchema, CreateMs365Method
from wiki.integrations.ms.exceptions import Ms365BadFormat
from wiki.integrations.ms.utils import get_doc_retriever_client
from wiki.integrations.ms.views import (
    handle_from_url,
    create_cloud_page,
    handle_empty_doc,
    ALLOWED_MIME,
    handle_upload_doc,
    handle_finalize_upload,
)
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.features.get_features import get_wiki_features


@upgrade_legacy_api_exceptions
def create_ms365_page(user, user_auth, organization: BaseOrganization, data: CreatePageSchema) -> OperationResult:
    if not get_wiki_features().ms365:
        raise BadRequest('Feature is disabled')

    cloud_page_settings = data.cloud_page
    client = get_doc_retriever_client()

    with client.user_identity(user_auth.tvm2_user_ticket):
        if cloud_page_settings.method == CreateMs365Method.FROM_URL:
            cloud_result, presentation_params = handle_from_url(client, data.cloud_page.url)

            result = create_page(user, user_auth, organization, data)
            create_cloud_page(result.page, cloud_result, presentation_params)
            return result

        elif cloud_page_settings.method == CreateMs365Method.EMPTY_DOC:
            cloud_result = handle_empty_doc(client, data.slug, data.cloud_page.doctype)

            result = create_page(user, user_auth, organization, data)
            create_cloud_page(result.page, cloud_result)
            return result

        elif cloud_page_settings.method == CreateMs365Method.UPLOAD_DOC:
            if data.cloud_page.mimetype not in ALLOWED_MIME:
                raise Ms365BadFormat()

            session_guid, upload_to = handle_upload_doc(
                client, data.slug, ALLOWED_MIME[data.cloud_page.mimetype], data.dict()
            )
            return OperationResult(
                page_created=False, response=Ms365UploadSessionSchema(upload_to=upload_to, upload_session=session_guid)
            )

        elif cloud_page_settings.method == CreateMs365Method.FINALIZE_UPLOAD:
            cloud_result, validated_data = handle_finalize_upload(client, data.cloud_page.upload_session)

            result = create_page(user, user_auth, organization, data)
            create_cloud_page(result.page, cloud_result)
            return result

        else:
            raise BadRequest()
