from django.db import transaction

from wiki.api_frontend.views.access import validate_acl_request, apply_legacy_acl, apply_legacy_acl_sideeffects
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import BadRequest
from wiki.api_v2.public.pages.exceptions import SlugOccupied, SlugReserved
from wiki.api_v2.public.pages.page.create.create_grid import create_grid
from wiki.api_v2.public.pages.page.create.create_ms365 import create_ms365_page
from wiki.api_v2.public.pages.page.create.create_regular_page import create_page
from wiki.api_v2.public.pages.page.details_serializer import PageCompositeSerializer
from wiki.api_v2.public.pages.page.permissions import assert_has_access_to_create_page
from wiki.api_v2.public.pages.page.schemas import CreatePageSchema
from wiki.api_v2.public.pages.page.sideeffects import apply_sideeffects_on_page_creation
from wiki.api_v2.public.pages.suggest_slug_view import is_slug_occupied
from wiki.api_v2.public.pages.utils import make_redlock_tag
from wiki.pages.access.consts import LegacyAcl
from wiki.pages.access.utils import convert_acl_request_to_legacy_acl
from wiki.pages.constants import ReservedSupertagAction
from wiki.pages.models.consts import PageType, AclType
from wiki.pages.reserved_supertags import is_reserved_supertag
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.db import on_commit
from wiki.utils.django_redlock.redlock import RedisLock
from wiki.utils.limits import assert_has_quota


FACTORIES = {
    PageType.REGULAR_PAGE: create_page,
    PageType.WYSIWYG: create_page,
    PageType.GRID: create_grid,
    PageType.CLOUD_PAGE: create_ms365_page,
}


@di
@legacy_org_ctx
def create_page_view(
    request, organization: BaseOrganization, data: CreatePageSchema, fields: str = None, is_silent: bool = False
):
    log_slug(request, slug=data.slug)

    desired_acl = None
    return_fields = PageCompositeSerializer.extract_fields(fields)

    with RedisLock(make_redlock_tag('create_page', organization, data.slug)):

        if is_slug_occupied(organization, data.slug):
            raise SlugOccupied()

        if is_reserved_supertag(data.slug, action=ReservedSupertagAction.CREATE):
            raise SlugReserved()

        assert_has_access_to_create_page(request.user, organization, slug=data.slug)
        assert_has_quota(organization)

        if data.acl:
            current_acl = LegacyAcl(acl_type=AclType.CUSTOM, is_readonly=False, break_inheritance=False)

            desired_acl = convert_acl_request_to_legacy_acl(data.acl)
            validate_acl_request(request.user, current_acl, desired_acl)

        if data.page_type not in FACTORIES:
            raise BadRequest(f'Creation of {data.page_type} is not implemented yet')

        # atomic_retry?
        with transaction.atomic():
            result = FACTORIES[data.page_type](request.user, request.user_auth, organization, data)

            if not result.page_created:
                return result.response

            page = result.page
            revision = result.revision

            if desired_acl:
                apply_legacy_acl(page, request.user, desired_acl)
                apply_legacy_acl_sideeffects(page)

            on_commit(
                lambda: apply_sideeffects_on_page_creation(
                    page, revision, is_silent=is_silent, subscribe_me=data.subscribe_me
                )
            )

    return PageCompositeSerializer.serialize_page(page, return_fields, request, organization)
