from typing import List

from wiki.api_v2.collections import CompositeSerializer
from wiki.api_v2.public.pages.page.getters import (
    get_authors_schema,
    get_background,
    get_content,
    get_last_revision_id,
    get_page_attributes_schema,
    get_bookmark_schema,
    get_subscription_schema,
    get_redirect_schema,
    get_cluster_schema_from_page,
)
from wiki.api_v2.public.pages.page.permissions import get_user_permissions_for_page
from wiki.api_v2.public.pages.schemas import (
    PageFullDetailsSchema,
)
from wiki.pages.logic.acl import get_acl_preview, get_acl_schema
from wiki.pages.logic.actuality import get_actuality
from wiki.pages.logic.breadcrumbs import get_breadcrumbs_from_page
from wiki.pages.logic.officiality import get_officiality
from wiki.pages.models import Page


class PageCompositeSerializer(CompositeSerializer[Page, PageFullDetailsSchema]):
    # field in schema -> (read, [deps]])
    FIELDS = {
        'redirect': (get_redirect_schema, []),
        'cluster': (get_cluster_schema_from_page, []),
        'authors': (get_authors_schema, []),
        'background': (get_background, []),
        'breadcrumbs': (get_breadcrumbs_from_page, []),
        'actuality': (get_actuality, []),
        'attributes': (get_page_attributes_schema, []),
        'access': (get_acl_preview, []),
        'acl': (get_acl_schema, ['organization']),
        'content': (get_content, ['request', 'revision']),
        'last_revision_id': (get_last_revision_id, []),
        'officiality': (get_officiality, []),
        'bookmark': (get_bookmark_schema, ['request']),
        'subscription': (get_subscription_schema, ['request']),
        'user_permissions': (get_user_permissions_for_page, ['organization', 'user']),
    }

    @classmethod
    def serialize_page(
        cls, page: Page, fields: List[str], request, organization, revision=None
    ) -> PageFullDetailsSchema:
        base = PageFullDetailsSchema.serialize(page, revision)
        return cls._add_fields(
            base,
            page,
            fields,
            {'request': request, 'user': request.user, 'organization': organization, 'revision': revision},
        )
