from typing import Union
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404

from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.exceptions import RedirectChain
from wiki.api_v2.public.pages.page.schemas import RedirectChainSchema
from wiki.api_v2.public.pages.page_identity import PageIdentity
from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.public.pages.page.details_serializer import PageCompositeSerializer
from wiki.api_v2.public.pages.schemas import PageFullDetailsSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.pages.logic.etalon import get_etalon_page
from wiki.pages.models import RedirectLoopException
from wiki.personalisation.user_cluster import expand_user_cluster_alias
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.supertag import normalize_supertag


def _build_response(
    request,
    organization: BaseOrganization,
    page,
    raise_on_redirect: bool = False,
    revision_id: int = None,
    fields: str = None,
) -> Union[PageFullDetailsSchema, HttpResponseRedirect]:
    # редиректы отрабатывают до проверки доступа. но если пойти за контентом редиректа, проверка все же будет
    if page.has_redirect() and raise_on_redirect:
        target = page
        try:
            target = page.redirect_target()
        except RedirectLoopException:
            pass

        raise RedirectChain(
            details=RedirectChainSchema(
                source_page=PageIdentity.serialize(page), target_page=PageIdentity.serialize(target)
            ).dict()
        )
    assert_has_access(request.user, organization, page, Action.READ)
    revision = None
    if revision_id:
        revision = get_object_or_404(page.revision_set.all(), id=revision_id)

    fields = PageCompositeSerializer.extract_fields(fields)
    return PageCompositeSerializer.serialize_page(page, fields, request, organization, revision)


@di
@legacy_org_ctx
def get_page_details_view(
    request,
    organization: BaseOrganization,
    slug: str,
    raise_on_redirect: bool = False,
    revision_id: int = None,
    fields: str = None,
) -> Union[PageFullDetailsSchema, HttpResponseRedirect]:
    """
    Возвращает детали страницы
    - `slug` - ненормализованный slug страницы, например users/something/абв
    - `fields` - желаемые дополнительные блоки, через запятую; см. описание формата ответа за деталями
    - `raise_on_redirect` - если на странице установлен редирект, будет взведена ошибка
    """

    slug = expand_user_cluster_alias(slug, request.user)
    slug = normalize_supertag(slug)

    try:
        page = get_page_or_404(organization, supertag=slug)
    except IsStubPage:
        page = get_etalon_page(organization, slug=slug)

    log_slug(request, slug=page.slug)
    return _build_response(request, organization, page, raise_on_redirect, revision_id, fields)


@di
@legacy_org_ctx
def get_page_details_by_id_view(
    request,
    organization: BaseOrganization,
    idx: int,
    raise_on_redirect: bool = False,
    revision_id: int = None,
    fields: str = None,
) -> Union[PageFullDetailsSchema, HttpResponseRedirect]:
    """
    Возвращает детали страницы:
    - `fields` - желаемые дополнительные блоки, см. описание формата ответа за деталями
    - `follow_redirects` - если на странице установлен редирект, будет совершен редирект на details целевой страницы
    """
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    return _build_response(request, organization, page, raise_on_redirect, revision_id, fields)
