from django.conf import settings

from wiki.pages.models import AccessRequest, Revision
from wiki.acl.check_access import filter_by_access
from wiki.api_frontend.views.grids import get_grid_data_api_v2
from wiki.api_v2.exceptions import FeatureNotImplemented
from wiki.api_v2.public.pages.consts import SubscriptionStatus
from wiki.api_v2.public.pages.page.preserve_order import restore_order
from wiki.api_v2.public.pages.schemas import (
    AuthorSchema,
    BackgroundSchema,
    ClusterSchema,
    CloudPageContentSchema,
    MovedOutPage,
    PageAttributesSchema,
    PageDetailsSchema,
    RedirectSchema,
    PageDetailsBookmarkSchema,
    PageDetailsSubscriptionSchema,
    AccessRequestSchema,
)
from wiki.api_v2.schemas import Slug
from wiki.favorites.models import Bookmark
from wiki.grids.consts import GridContentSchema
from wiki.grids.models import Grid
from wiki.integrations import get_page_provider
from wiki.pages.models import Page, CloudPage, LocationHistory
from wiki.subscriptions.logic import get_user_page_subscriptions
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.user_data_repository import USER_DATA_REPOSITORY

EXCLUDED_SLUGS = settings.NAVIGATION_TREE_EXCLUDED_SLUGS
CLUSTER_LEVEL = settings.NAVIGATION_TREE_CLUSTER_LEVEL


def get_cluster_schema_from_page(page: Page) -> ClusterSchema | None:
    # Простейшая логика определения кластера для страницы.
    # Кластер расчитывается исходя из параметра NAVIGATION_TREE_CLUSTER_LEVEL
    # Для интранета это раздел первого уровня для всех страниц
    # за исключением users - там кластеры - users/{username})

    return get_cluster_schema_from_slug(page.slug)


def get_cluster_schema_from_slug(slug: Slug = None) -> ClusterSchema | None:
    if slug in EXCLUDED_SLUGS:
        return None

    parts = slug.split('/')
    cluster = '/'.join(parts[:CLUSTER_LEVEL])

    if cluster in EXCLUDED_SLUGS:
        cluster = '/'.join(parts[:2])

    return ClusterSchema(slug=cluster)


def get_authors_schema(page: Page) -> AuthorSchema:
    all_authors = restore_order(page, 'authors', page.get_authors())
    owner = page.owner or page.last_author or all_authors[0]

    # На переходный период, owner может быть вообще не относящимся к странице
    # Поэтому если вдруг это так, отдаем кого-то из настоящих авторов в качестве главного

    if owner not in all_authors:
        owner = all_authors[0]

    return AuthorSchema(
        owner=USER_DATA_REPOSITORY.orm_to_user_schema(owner),
        last_author=USER_DATA_REPOSITORY.orm_to_user_schema(page.last_author),
        all=USER_DATA_REPOSITORY.orm_to_user_schema(all_authors),
    )


def get_content(page: Page, request, revision) -> CloudPageContentSchema | GridContentSchema | str:
    if page.page_type in (page.TYPES.PAGE, page.TYPES.WYSIWYG):
        if revision:
            return revision.body
        return page.body

    if page.page_type == page.TYPES.CLOUD:
        cloud_page = CloudPage.objects.get(page_id=page.id)
        page_provider = get_page_provider(cloud_page.provider)
        return page_provider.to_apiv2_view(cloud_page)

    if page.page_type == page.TYPES.GRID:
        return get_grid_data_api_v2(Grid.objects.get(pk=page.pk), request, revision)

    raise FeatureNotImplemented()


def get_last_revision_id(page: Page) -> int | None:
    return Revision.objects.filter(page_id=page.id).order_by('-created_at', '-id').values_list('id', flat=True).first()


def get_redirect_schema(page: Page) -> RedirectSchema | None:
    if not page.has_redirect():
        return None

    return RedirectSchema(
        page_id=page.redirects_to_id, redirect_target=PageDetailsSchema.serialize(page.redirect_target())
    )


def get_page_attributes_schema(page: Page) -> PageAttributesSchema:
    return PageAttributesSchema(
        created_at=page.created_at,
        modified_at=page.modified_at,
        lang=page.lang,
        is_readonly=page.is_readonly,
        comments_count=page.comments_count,
        comments_enabled=page.is_comments_enabled,
        keywords=page.keywords_list,
    )


def get_bookmark_schema(page: Page, request) -> PageDetailsBookmarkSchema | None:
    bms = Bookmark.objects.filter(page=page, user=request.user).values_list('pk', flat=True)

    if len(bms) == 0:
        return None

    return PageDetailsBookmarkSchema(id=bms[0])


def get_subscription_schema(page: Page, request) -> PageDetailsSubscriptionSchema | None:
    subscription = get_user_page_subscriptions(user=request.user, page=page)
    if subscription is None:
        return None

    status = SubscriptionStatus.CLUSTER if subscription.is_cluster else SubscriptionStatus.PAGE
    return PageDetailsSubscriptionSchema(status=status)


def get_moved_out_pages_for_slug(slug, user, org: BaseOrganization) -> list[MovedOutPage]:
    location_histories = LocationHistory.objects.prefetch_related('page').filter(
        slug=slug,
        page__status__gt=0,
        page__org=org.as_django_model(),
    )
    pages = [location_history.page for location_history in location_histories]
    accessible_pages = set(filter_by_access(pages, user=user, organization=org))
    return [MovedOutPage.serialize(page, accessible=page in accessible_pages) for page in pages]


def get_background(page: Page) -> BackgroundSchema | None:
    background = settings.PAGE_BACKGROUNDS_IDS.get(page.background_id, None)
    return background and BackgroundSchema.parse_obj(background)


def get_pending_access_request(page: Page, user) -> AccessRequestSchema | None:
    qs = AccessRequest.objects.filter(applicant_id=user.id, page_id=page.id, verdict_by=None).first()
    return AccessRequestSchema(reason=qs.reason, requested_at=qs.created_at) if qs else None
