from django.shortcuts import get_object_or_404
from django.db import transaction

from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.public.pages.exceptions import SlugOccupied, SlugReserved
from wiki.api_v2.public.pages.page.create_view import assert_has_access_to_create_page, assert_has_quota
from wiki.api_v2.public.pages.page.details_serializer import PageCompositeSerializer
from wiki.api_v2.public.pages.page.schemas import PageImportSchema
from wiki.api_v2.public.pages.page.sideeffects import apply_sideeffects_on_page_creation
from wiki.api_v2.public.pages.suggest_slug_view import is_slug_occupied
from wiki.api_v2.public.pages.utils import make_redlock_tag
from wiki.pages.constants import ReservedSupertagAction
from wiki.pages.logic.import_file import import_file_from_upload_session
from wiki.pages.models import Page, Revision
from wiki.pages.models.consts import DB_PAGE_TYPES
from wiki.pages.reserved_supertags import is_reserved_supertag
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.uploads.models import UploadSession
from wiki.utils import timezone
from wiki.utils.db import on_commit
from wiki.utils.django_redlock.redlock import RedisLock


@di
@legacy_org_ctx
def page_import(request, organization: BaseOrganization, data: PageImportSchema, fields: str = None):
    log_slug(request, slug=data.slug)
    with RedisLock(make_redlock_tag('import_page', organization, data.slug)):

        if is_slug_occupied(organization, data.slug):
            raise SlugOccupied()

        if is_reserved_supertag(data.slug, action=ReservedSupertagAction.CREATE):
            raise SlugReserved()

        assert_has_access_to_create_page(request.user, organization, data.slug)
        assert_has_quota(organization)

        with transaction.atomic():
            upload_session: UploadSession = get_object_or_404(
                UploadSession.objects.select_for_update(),
                session_id=data.upload_session,
                user=request.user,
                org=organization.as_django_model(),
            )

            page = Page(
                title=data.title,
                page_type=DB_PAGE_TYPES.PAGE,
                tag=data.slug,
                supertag=data.slug,
                owner=request.user,
                org=organization.as_django_model(),
                last_author=request.user,
                modified_at=timezone.now(),
            )
            page.body = ''
            page.save()
            page.authors.add(request.user)
            revision = Revision.objects.create_from_page(page)

            import_file_from_upload_session(upload_session, page, request.user)

            on_commit(lambda: apply_sideeffects_on_page_creation(page, revision, subscribe_me=data.subscribe_me))

    return_fields = PageCompositeSerializer.extract_fields(fields)
    return PageCompositeSerializer.serialize_page(page, return_fields, request, organization)
