from typing import List

from wiki.acl.check_access import _check_access
from wiki.acl.consts import Action
from wiki.api_v2.exceptions import Forbidden
from wiki.api_v2.public.pages.schemas import UserPermission
from wiki.api_v2.schemas import Slug
from wiki.pages.constants import ReservedSupertagAction
from wiki.pages.logic.hierarchy import get_nearest_existing_parent
from wiki.pages.models import Page
from wiki.pages.models.consts import COMMENTS_STATUS
from wiki.pages.reserved_supertags import is_reserved_supertag
from wiki.pages.access import is_admin


def has_access_to_create_page(user, organization, slug: Slug):
    if is_reserved_supertag(slug, action=ReservedSupertagAction.DELETE):
        return False

    nearest_parent = get_nearest_existing_parent(slug, organization=organization)
    if nearest_parent is None:
        return True
    # не опечатка, именно READ -- по крайней мере такая логика была в старой API
    return _check_access(user, organization, '', nearest_parent, Action.READ)


def assert_has_access_to_create_page(user, organization, slug: Slug):
    if not has_access_to_create_page(user, organization, slug=slug):
        raise Forbidden()


def has_access_to_process_access_requests(user, organization, page: Page) -> bool:
    return user in page.get_authors() or is_admin(user)

    # в старой апишке была еще проверка на то,
    # является ли человек в группах искался ответственный (responsibles)
    # код тут - но я преднамеренно его не включаю.
    # if not has_permission:
    #     responsible_ids = get_group_responsibles_for_page(page)
    #     return user.staff.pk in responsible_ids


UserPermissionToActionGeneral = {
    UserPermission.CAN_EDIT: Action.WRITE,
    UserPermission.CAN_CHANGE_ACL: Action.CHANGE_ACL,
    UserPermission.CAN_CHANGE_AUTHORS: Action.CHANGE_AUTHORS,
    UserPermission.CAN_DELETE: Action.DELETE,
}

UserPermissionToActionReserved = {
    UserPermission.CAN_EDIT: Action.WRITE,
    UserPermission.CAN_CHANGE_ACL: Action.CHANGE_ACL,
    UserPermission.CAN_CHANGE_AUTHORS: Action.CHANGE_AUTHORS,
}


# user, organization passed via argname, don't rename
def get_user_permissions_for_page(page, user, organization) -> List[UserPermission]:
    action_map = UserPermissionToActionGeneral
    if is_reserved_supertag(page.slug, action=ReservedSupertagAction.DELETE):
        action_map = UserPermissionToActionReserved

    perms = [k for k, v in action_map.items() if _check_access(user, organization, '', page, v)]
    if page.comments_status == COMMENTS_STATUS.enabled:
        perms.append(UserPermission.CAN_COMMENT)

    perms.append(UserPermission.CAN_VIEW)
    return perms


# user, org passed via argname, don't rename
def get_user_permissions_for_slug(slug, user, org) -> List[UserPermission]:
    perms = []
    if has_access_to_create_page(user, org, slug):
        perms.append(UserPermission.CAN_CREATE_PAGE)
    return perms
