from enum import Enum
from typing import Optional, List, Union, Literal, Final

from ninja import Schema
from pydantic import HttpUrl, Field, UUID4, conlist, root_validator, conint, constr

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.api_v2.public.pages.consts import Location
from wiki.api_v2.public.pages.page_identity import PageIdentity
from wiki.api_v2.schemas import Slug, NotEmptyStringTrimmed
from wiki.integrations.ms.consts import Ms365DocType
from wiki.pages.models.consts import AclType, ACL_TYPE_DESCRIPTION, PageType
from wiki.utils.features.get_features import get_wiki_features

NOT_SET: Final = object()


class RedirectChainSchema(Schema):
    source_page: PageIdentity
    target_page: PageIdentity


class ActualityUpdateSchema(Schema):
    is_actual: bool
    comment: Optional[str]

    external_links: Optional[List[HttpUrl]]
    actual_pages: Optional[List[PageIdentity]] = Field(
        description='Список либо `page_id` либо `slug` на актуальную страницу'
    )


def validate(cls, attrs):
    if attrs.get('comment'):
        attrs['comment'] = None

    return attrs


class AuthorUpdateSchema(Schema):
    owner: Optional[UserIdentity] = Field(
        description='Главный автор, чья иконка отображается на странице; Если среди всех его нет, будет добавлен'
    )
    all: Optional[conlist(UserIdentity, min_items=1, max_items=256)]


class RedirectUpdateSchema(Schema):
    page: Optional[PageIdentity] = Field(
        description='Страница, куда должен быть совершен редирект. `null` для снятия редиректа'
    )


class AclUpdateSchema(Schema):
    """
    Допустимые комбинации:
    0. { is_readonly: False/True }
    1. { break_inheritance: False, is_readonly: False/True/UNSET }
    2. { break_inheritance: True, acl_type: DEFAULT | ONLY_AUTHORS, is_readonly: False/True/UNSET }
    3. { break_inheritance: True, acl_type: CUSTOM, users: [], groups: [], departments: [],
    is_readonly: False/True/UNSET}
    """

    break_inheritance: Optional[bool] = Field(
        description='Если True - наследование от родителя отключено. Иначе  acl_type, users и groups не имеют '
        'эффекта, а все права наследуются от родителя',
        default_factory=lambda: NOT_SET,
    )

    is_readonly: Optional[bool] = Field(
        description='Если True - страница недоступна для редактирования всем кроме авторов вне зависимости от ACL',
        default_factory=lambda: NOT_SET,
    )

    acl_type: Optional[AclType] = Field(description=ACL_TYPE_DESCRIPTION, default_factory=lambda: NOT_SET)

    users: Optional[List[UserIdentity]] = Field(
        description='Для `AclType.CUSTOM` указывает пользователей которым доступна страница',
        default_factory=lambda: NOT_SET,
    )

    groups: Optional[List[str]] = Field(
        description='Для `AclType.CUSTOM` указывает группы которым доступна страница. Доступ выдается также для всех'
        ' подгрупп выбранных групп. Для внешнего инстанса - dir_id, для внутреннего - staff_id',
        default_factory=lambda: NOT_SET,
    )

    departments: Optional[List[str]] = Field(
        description='Для `AclType.CUSTOM` указывает группы которым доступна страница. Доступ выдается также для всех'
        ' подгрупп выбранных групп (для B2B, для интранета не используется)',
        default_factory=lambda: NOT_SET,
    )

    @property
    def affects_acl(self):
        return self.break_inheritance is not NOT_SET

    @property
    def affects_page_model(self):
        return self.is_readonly is not NOT_SET

    @root_validator
    def validation(cls, data):
        """
        Допустимые варианты:
           0. { is_readonly: False/True }
           1. { break_inheritance: False, is_readonly: False/True/UNSET }
           2. { break_inheritance: True, acl_type: DEFAULT | ONLY_AUTHORS, is_readonly: False/True/UNSET }
           3. { break_inheritance: True, acl_type: CUSTOM, users: [], groups [], is_readonly: False/True/UNSET }
        """

        msg_0 = 'For break_inheritance=false, other options affecting ACL must be omitted'
        msg_1 = 'For break_inheritance=true, acl_type is required'
        msg_2 = 'For custom ACL, either users, groups or departements is required'

        if data['break_inheritance'] is False:
            assert data['acl_type'] is NOT_SET, msg_0
            assert data['users'] is NOT_SET, msg_0
            assert data['groups'] is NOT_SET, msg_0
            assert data['departments'] is NOT_SET, msg_0

        if data['break_inheritance'] is True:
            assert data['acl_type'] is not NOT_SET, msg_1

        def _set_or_non_empty(lst):
            return lst is not NOT_SET and len(lst) > 0

        if data['acl_type'] == AclType.CUSTOM:
            assert (
                _set_or_non_empty(data['users'])
                or _set_or_non_empty(data['groups'])
                or _set_or_non_empty(data['departments'])
            ), msg_2

        return data


class PageUpdateSchema(Schema):
    title: NotEmptyStringTrimmed = Field(default_factory=lambda: NOT_SET)
    content: str = Field(default_factory=lambda: NOT_SET)
    redirect: RedirectUpdateSchema = Field(default_factory=lambda: NOT_SET)
    authors: AuthorUpdateSchema = Field(default_factory=lambda: NOT_SET)
    acl: AclUpdateSchema = Field(default_factory=lambda: NOT_SET)
    actuality: ActualityUpdateSchema = Field(default_factory=lambda: NOT_SET)
    background: Optional[conint(ge=0)] = Field(
        default_factory=lambda: NOT_SET, description='Id подложки. `null` для снятия'
    )

    # attributes: Optional[PageAttributesSchema] = _mk_hint('attributes')
    # content: Optional[Union[CloudPageContentSchema, GridContentSchema, str]] = _mk_hint('content', CONTENT_DESC)
    # officiality: Optional[OfficialitySchema] = _mk_hint('officiality')


class PageAppendContentBodySchema(Schema):
    location: Location


class PageAppendContentSectionSchema(Schema):
    id: int
    location: Location


class PageAppendContentAnchorSchema(Schema):
    name: str


class PageAppendContentSchema(Schema):
    content: constr(min_length=1)
    body: Optional[PageAppendContentBodySchema]
    section: Optional[PageAppendContentSectionSchema]
    anchor: Optional[PageAppendContentAnchorSchema]

    @root_validator
    def validation(cls, data):
        """Должно быть заполнено только одно из полей: 'body', 'section', 'anchor'"""

        mutually_exclusive = ('body', 'section', 'anchor')
        msg = f'Fields {mutually_exclusive} are mutually exclusive'
        fields_count = [data.get(field) is not None for field in mutually_exclusive].count(True)
        assert fields_count == 1, msg
        return data


class CreateMs365Method(str, Enum):
    FROM_URL = 'from_url'
    EMPTY_DOC = 'empty_doc'
    UPLOAD_DOC = 'upload_doc'
    FINALIZE_UPLOAD = 'finalize_upload'


class NewMs365(Schema):
    method: Literal[CreateMs365Method.EMPTY_DOC]
    doctype: Ms365DocType


class ImportMs365FromLink(Schema):
    method: Literal[CreateMs365Method.FROM_URL]
    url: str


class UploadMs365(Schema):
    method: Literal[CreateMs365Method.UPLOAD_DOC]
    mimetype: str


class FinalizeUploadMs365(Schema):
    method: Literal[CreateMs365Method.FINALIZE_UPLOAD]
    upload_session: str


SOURCE_TYPE_TO_MODEL = {
    CreateMs365Method.FINALIZE_UPLOAD: FinalizeUploadMs365,
    CreateMs365Method.UPLOAD_DOC: UploadMs365,
    CreateMs365Method.FROM_URL: ImportMs365FromLink,
    CreateMs365Method.EMPTY_DOC: NewMs365,
}


class Ms365UploadSessionSchema(Schema):
    upload_to: str
    upload_session: str


CLOUD_PAGE_DESCRIPTION = """Требуется для создания страницы с типом `cloud_page`'
  - Из урла (ссылка на Sharepoint или iframe c эмбеддом):
      - Используйте тип `ImportMs365FromLink` и `cloud_page.source` = `from_url`
  - Из нового документа:
      - Используйте тип `NewMs365` и `cloud_page.source` = 'empty_doc'
  - Загрузить с машины.
      - Шаг 1. Используйте тип `UploadMs365` и `cloud_page.source` = 'upload_doc'
      - Шаг 2. Загрузите через PUT по урлу upload_to
      - Шаг 3. Используйте тип FinalizeUploadMs365 и `cloud_page.source` = 'finalize_upload'
"""


class CreatePageSchema(Schema):
    page_type: PageType
    title: NotEmptyStringTrimmed
    slug: Slug

    cloud_page: Optional[Union[NewMs365, ImportMs365FromLink, UploadMs365, FinalizeUploadMs365]] = Field(
        description=CLOUD_PAGE_DESCRIPTION, discriminator='method'
    )

    content: Optional[str]

    subscribe_me: bool = Field(default=False, description='Подписатьcя на изменения')

    acl: Optional[AclUpdateSchema] = Field(
        description='Если не указан, будет новая страница будет наследовать права доступа'
    )

    @root_validator
    def validation(cls, data):
        if data.get('page_type') == PageType.CLOUD_PAGE:
            assert get_wiki_features().ms365, 'Feature is disabled'
            assert data.get('cloud_page'), 'Please provide cloud_page field'

        return data


class LinkResponse(Schema):
    url: str


class PageImportSchema(Schema):
    title: NotEmptyStringTrimmed
    slug: Slug
    subscribe_me: bool = Field(default=False, description='Подписатьcя на изменения')
    upload_session: UUID4
