from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.exceptions import BadRequest
from wiki.api_v2.public.pages.page.preserve_order import store_order
from wiki.api_v2.public.pages.page.schemas import AuthorUpdateSchema
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.dao import get_user_by_identity, get_users_by_identity, UserNotFound


def update_authors_schema(
    user, organization: BaseOrganization, page: Page, data: AuthorUpdateSchema, is_silent: bool = False
):
    assert_has_access(
        user, organization, page, Action.CHANGE_AUTHORS, message='Authors change requires elevated' ' permissions'
    )

    old_authors = list(page.get_authors())
    old_owner = page.owner
    new_owner = old_owner

    new_authors = old_authors.copy()
    try:
        if data.owner:
            new_owner = get_user_by_identity(data.owner, organization=organization)
        if data.all:
            new_authors = get_users_by_identity(
                data.all, organization=organization, apiv1_exceptions=False, preserve_order=True
            )
            store_order(page, 'authors', new_authors)
    except UserNotFound as u:
        raise BadRequest(
            'Some of requested users not found',
            details={
                'missing': [m.dict(exclude_unset=True) for m in u.missing_identities],
            },
        )

    if new_owner and new_owner not in new_authors:
        new_authors.append(new_owner)

    if new_owner and new_owner != old_owner:
        page.owner = new_owner
        page.save()

    if set(new_authors) != set(old_authors):
        # мы не храним персистентно порядок авторов, но для фронта важно чтобы после update
        # порядок не сильно изменился в ответе
        store_order(page, 'authors', new_authors)
        page.authors.set(new_authors)
