import logging

from django.conf import settings

from wiki.api_v2.exceptions import FeatureNotImplemented
from wiki.api_v2.public.pages.exceptions import PageBodySizeExceeded
from wiki.api_v2.public.pages.page.sideeffects import apply_sideeffects_on_page_edit
from wiki.pages.models import Page, Revision
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils import timezone
from wiki.utils.db import on_commit

logger = logging.getLogger(__name__)


def update_page_content(user, organization: BaseOrganization, page: Page, data: str, is_silent: bool = False):
    if page.page_type not in (Page.TYPES.PAGE, Page.TYPES.WYSIWYG):
        raise FeatureNotImplemented(debug_message='Direct content edition is allowed only for Page with type page')

    if page.body == data:
        return

    if limit := settings.LIMIT__WIKI_TEXT_FOR_PAGE__BYTES:
        body_size = len(data.encode('utf-8'))
        if body_size > limit:
            logger.warning(f'Page {page.supertag}, body is too big "{body_size}"')
            raise PageBodySizeExceeded(debug_message=f"Max body size is {limit} bytes")

    page.body = data
    page.last_author = user
    page.modified_at = timezone.now()
    page.save()

    revision = Revision.objects.create_from_page(page)

    on_commit(lambda: apply_sideeffects_on_page_edit(page, revision, is_silent=is_silent))
