from datetime import timedelta

from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.favorites_v2.logic import update_autofolders
from wiki.grids.models import Revision
from wiki.integrations.ms.wiki_to_retriever_sqs import RETRIEVER_SQS_CLIENT
from wiki.notifications.models import PageEvent
from wiki.pages.api import subscribe_to_new_page, create_event
from wiki.pages.models import Page
from wiki.pages.signals import access_changed
from wiki.utils import timezone
from wiki.utils.tasks import TrackLinksTask
from wiki.pages.signals import access_changed as access_changed_signal


def apply_sideeffects_on_page_creation(
    page: Page,
    revision: Revision,
    is_silent: bool = False,
    subscribe_me: bool = False,
):
    """
    Обязательно выполнять в on_commit()
    """
    # Cloudsearch
    CLOUD_SEARCH_CLIENT.on_model_upsert(page)
    # Подписать всех кто наблюдает за кластером (старая схема) и возможно самого автора
    subscribe_to_new_page(page, subscribe_author=subscribe_me)
    # Добавить событие
    create_event(True, page, revision, 20, notify=not is_silent)
    # Обновить Папки "я автор" и прочее
    update_autofolders(page, was_created=True)
    # Сигналы-сигнальчики
    access_changed.send(sender=None, page_list=[page])
    # Обновить ссылки от страницы
    if page.page_type in (Page.TYPES.GRID, Page.TYPES.PAGE):
        TrackLinksTask().delay(page.id, is_grid=page.page_type == Page.TYPES.GRID, is_new_page=True)


def apply_sideeffects_on_page_edit(page: Page, revision: Revision, is_silent: bool = False):
    """
    Обязательно выполнять в on_commit()
    """
    # Cloudsearch
    CLOUD_SEARCH_CLIENT.on_model_upsert(page)
    # Добавить событие
    create_event(False, page, revision, 20, notify=not is_silent)
    # Обновить Папки "я автор" и прочее
    update_autofolders(page, was_created=False)
    # Обновить ссылки от страницы
    if page.page_type in (Page.TYPES.GRID, Page.TYPES.PAGE):
        TrackLinksTask().delay(page.id, is_grid=page.page_type == Page.TYPES.GRID, is_new_page=False)


def apply_sideeffects_on_page_delete(page: Page, user):
    """
    Необязательно выполнять в on_commit()
    """

    CLOUD_SEARCH_CLIENT.on_model_delete(page)
    RETRIEVER_SQS_CLIENT.on_model_delete(page)

    PageEvent(
        event_type=PageEvent.EVENT_TYPES.delete,
        page=page,
        author=user,
        timeout=timezone.now() + timedelta(minutes=5),
        meta={},
    ).save()

    access_changed_signal.send(sender=None, page_list=[page.supertag])  # сбросить кэш по сигналу
