from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.page.details_serializer import PageCompositeSerializer
from wiki.api_v2.public.pages.page.schemas import PageUpdateSchema, NOT_SET
from wiki.api_v2.public.pages.page.set_acl import set_acl
from wiki.api_v2.public.pages.page.set_actuality import update_actuality_schema
from wiki.api_v2.public.pages.page.set_authors import update_authors_schema
from wiki.api_v2.public.pages.page.set_background import update_background
from wiki.api_v2.public.pages.page.set_content import update_page_content
from wiki.api_v2.public.pages.page.set_redirect import update_redirect_schema
from wiki.api_v2.public.pages.schemas import PageFullDetailsSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.pages.logic.etalon import create_etalon_page
from wiki.pages.models import Page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils import timezone

WRITE_OPTIONAL_FIELDS = {
    'redirect': (update_redirect_schema, []),
    'acl': (set_acl, []),
    'authors': (update_authors_schema, []),
    'actuality': (update_actuality_schema, []),
    'background': (update_background, []),
    'content': (update_page_content, []),
}


@di
@legacy_org_ctx
def update_page_details(
    request,
    organization: BaseOrganization,
    idx: int,
    data: PageUpdateSchema,
    fields: str = None,
    is_silent: bool = False,
) -> PageFullDetailsSchema:
    """
    - fields - желаемые дополнительные блоки, которые должны вернуться. Смотри описание формата ответа за деталями
    """

    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.WRITE)

    _update_page_attrs(request.user, organization, page, data)

    for field_name, (fn, deps) in WRITE_OPTIONAL_FIELDS.items():
        val = getattr(data, field_name)
        if val is NOT_SET:
            continue
        fn(request.user, organization, page, val, is_silent)

    fields = PageCompositeSerializer.extract_fields(fields)
    page.modified_at = timezone.now()
    page.save()

    return PageCompositeSerializer.serialize_page(page, fields, request, organization)


def _update_page_attrs(user, organization: BaseOrganization, page: Page, data: PageUpdateSchema):
    changed = False
    if data.title is not NOT_SET:
        page.title = data.title
        changed = True

    return changed
