from typing import List
from typing import Optional

from django.db.models import Q
from ninja import Schema
from pydantic import root_validator, Field, PydanticValueError

from wiki.api_v2.exceptions import BadRequest
from wiki.api_v2.schemas import Slug
from wiki.sync.connect.base_organization import BaseOrganization


class PageIdentityError(PydanticValueError):
    code = 'page_identity'
    msg_template = 'must pass either id or slug'


class PageIdentity(Schema):
    id: Optional[int] = Field(
        description='Обязателен либо `id` либо `slug`, при передаче обоих `id` имеет приоритет ' 'и `slug` игнорируется'
    )
    slug: Optional[Slug]

    @root_validator
    def validator(cls, data):
        id_passed = bool(data.get('id', None))
        slug_passed = bool(data.get('slug', None))
        if not id_passed and not slug_passed:  # xor
            raise PageIdentityError()

        return data

    @classmethod
    def serialize(cls, page):
        return cls(id=page.id, slug=page.slug)


def resolve_page_identity(organization: BaseOrganization, page: PageIdentity):
    return resolve_page_identities(organization, [page])[0]


def resolve_page_identities(organization: BaseOrganization, actual_pages: List[PageIdentity], only_active: bool = True):
    ids = [q.id for q in actual_pages if q.id is not None]
    slugs = [q.slug for q in actual_pages if q.slug is not None and q.id is None]
    query = None

    if ids:
        query = Q(pk__in=ids)

    if slugs:
        q = Q(supertag__in=slugs)
        if query is None:
            query = q
        else:
            query = query | q

    if only_active:
        resolved_pages = organization.get_active_pages().filter(query)
    else:
        resolved_pages = organization.get_pages().filter(query)

    if len(resolved_pages) != len(actual_pages):
        resolved_pks = {p.id for p in resolved_pages}
        resolved_slugs = {p.slug for p in resolved_pages}

        missing = [PageIdentity(id=idx) for idx in (set(ids) - resolved_pks)] + [
            PageIdentity(slug=slug) for slug in (set(slugs) - resolved_slugs)
        ]

        raise BadRequest(
            'Some PageIdentities were not resolved',
            details={
                'missing': [m.dict(exclude_unset=True) for m in missing],
            },
        )

    return resolved_pages
