from datetime import timedelta
from typing import Tuple, Optional

from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.di import di, log_slug, legacy_org_ctx
from wiki.api_v2.exceptions import IsStubPage
from wiki.api_v2.public.pages.schemas import (
    RequestAuthorRoleResponseSchema,
    RequestAuthorRoleSchema,
    RequestAuthorRoleResult,
    RequestAuthorRoleDetails,
    GrantAuthorRoleSchema,
)
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.notifications.signals import request_page_author_created
from wiki.pages.access import is_admin
from wiki.pages.logic.authors import add_user_to_authors
from wiki.pages.logic.etalon import create_etalon_page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.dao import get_user_by_identity
from wiki.utils import timezone


def is_applicable_for_autogrant(user, page) -> Tuple[bool, Optional[RequestAuthorRoleDetails]]:
    if is_admin(user):
        return True, RequestAuthorRoleDetails.ADMIN

    if page.is_all_authors_dismissed():
        return True, RequestAuthorRoleDetails.EVERYONE_DISMISSED

    for parent_page in page.ancestors:
        if parent_page.supertag != 'users':
            if user in parent_page.get_authors():
                return True, RequestAuthorRoleDetails.VIA_INHERITANCE
    return False, None


@di
@legacy_org_ctx
def request_author_role_view(
    request,
    organization: BaseOrganization,
    idx: int,
    data: RequestAuthorRoleSchema,
) -> RequestAuthorRoleResponseSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    if request.user in page.get_authors():
        return RequestAuthorRoleResponseSchema(result=RequestAuthorRoleResult.ALREADY_AUTHOR)

    auto_grant, auto_grant_reason = is_applicable_for_autogrant(request.user, page)

    if auto_grant:
        add_user_to_authors(page, request.user, request.user)
        return RequestAuthorRoleResponseSchema(result=RequestAuthorRoleResult.ROLE_GRANTED, details=auto_grant_reason)

    if PageEvent.objects.filter(
        page=page,
        event_type=EventTypes.request_page_author,
        author=request.user,
        created_at__gt=timezone.now() - timedelta(days=3),
    ).exists():
        return RequestAuthorRoleResponseSchema(result=RequestAuthorRoleResult.REQUEST_ALREADY_SENT)

    request_page_author_created(page, request.user, data.reason or '')

    return RequestAuthorRoleResponseSchema(result=RequestAuthorRoleResult.REQUEST_SENT)


@di
@legacy_org_ctx
def grant_author_role_view(
    request,
    organization: BaseOrganization,
    idx: int,
    data: GrantAuthorRoleSchema,
) -> RequestAuthorRoleResponseSchema:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.CHANGE_AUTHORS)

    new_author = get_user_by_identity(data.user, organization=organization)
    add_user_to_authors(page, request.user, new_author)

    return RequestAuthorRoleResponseSchema(result=RequestAuthorRoleResult.ROLE_GRANTED)
