import logging

from ninja import Query

from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.collections import Collection, PaginationQuery, CollectionFactory, OrderDirection
from wiki.api_v2.di import di, log_slug, legacy_org_ctx
from wiki.api_v2.exceptions import NotFound, IsStubPage
from wiki.api_v2.public.pages.exceptions import DiffNotAvailable
from wiki.api_v2.public.pages.revisions.schemas import (
    PageDiffSchema,
    WysiwygDiffSchema,
    GridDiffSchema,
    DiffSchema,
)
from wiki.api_v2.public.pages.schemas import RevisionSchema
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.grids.logic import diff
from wiki.legacy import json
from wiki.pages.logic.etalon import get_etalon_page
from wiki.pages.models import Page, Revision
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.utils.diff import DifflibDiff

logger = logging.getLogger(__name__)

diff_chunks = DifflibDiff(return_type='chunks')
diff_html = DifflibDiff(return_type='html')


def serialize_list(r: Revision) -> RevisionSchema:
    return RevisionSchema(id=r.id, author=USER_DATA_REPOSITORY.orm_to_user_schema(r.author), created_at=r.created_at)


@di
@legacy_org_ctx
def revisions_view(
    request, organization: BaseOrganization, idx: int, pagination: PaginationQuery = Query(...)
) -> Collection[RevisionSchema]:
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    return (
        CollectionFactory()
        .default_ordering(['created_at'], direction=OrderDirection.DESC)
        .ordered_build(
            qs=page.revision_set.select_related('author'),
            serializer=serialize_list,
            pagination=pagination,
        )
    )


@di
@legacy_org_ctx
def diff_view(request, organization: BaseOrganization, idx: int, revision_a: int, revision_b: int):
    """
    Расчитывает дифф между текущей ревизией и ревизией с id переданной в compare_against
    """
    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    revisions = list(page.revision_set.filter(id__in=[revision_a, revision_b]).order_by('created_at'))
    if len(revisions) != 2:
        raise NotFound('One of revisions was not found')

    old_rev, new_rev = revisions

    if page.page_type == Page.TYPES.PAGE:
        content_diff = PageDiffSchema(diff=diff_chunks.diff_texts(old_rev.body, new_rev.body))

    elif page.page_type == Page.TYPES.WYSIWYG:
        content_diff = WysiwygDiffSchema(diff=diff_chunks.diff_texts(old_rev.body, new_rev.body))
    elif page.page_type == Page.TYPES.GRID:
        grids_diff, titles_diff = diff.build_diff(
            old_grid=json.loads(old_rev.body),
            new_grid=json.loads(new_rev.body),
        )
        content_diff = GridDiffSchema(diff=grids_diff, column_titles=titles_diff)
    else:
        raise DiffNotAvailable()

    return DiffSchema(title=diff_html.diff_texts(old_rev.title, new_rev.title), content=content_diff)
