from typing import Union

from wiki.api_v2.collections import Collection
from wiki.api_v2.public.pages.access_requests.router import inject_access_request_routes
from wiki.api_v2.public.pages.attachments.router import inject_attachment_routes
from wiki.api_v2.public.pages.attachments.views import download_by_filename_slug_pair_view
from wiki.api_v2.public.pages.autocomplete_view import AutoCompleteResponse, autocomplete_view
from wiki.api_v2.public.pages.backgrounds_view import BackgroundsResponse, backgrounds_view
from wiki.api_v2.public.pages.backlinks_view import backlinks_view
from wiki.api_v2.public.pages.comments.router import inject_comment_routes
from wiki.api_v2.public.pages.get_inherited_acl_view import get_inherited_acl
from wiki.api_v2.public.grids.schemas import PageGridsSchema
from wiki.api_v2.public.grids.views import page_grids_view
from wiki.api_v2.public.pages.homepage_view import HomepageResponse, homepage_view
from wiki.api_v2.public.pages.move_view import move_view
from wiki.api_v2.public.pages.ms365.check_access import check_m365_access_view
from wiki.api_v2.public.pages.ms365.download_source import download_ms365_source_view
from wiki.api_v2.public.pages.ms365.ensure_access import ensure_m365_access_view
from wiki.api_v2.public.pages.descendants_view import descendants_view
from wiki.api_v2.public.pages.navigation_tree_view import navigation_tree_view
from wiki.api_v2.public.pages.page.append_content_view import append_content_view
from wiki.api_v2.public.pages.page.clone_view import clone_page_view
from wiki.api_v2.public.pages.page.create_view import create_page_view
from wiki.api_v2.public.pages.page.delete_view import delete_page_view
from wiki.api_v2.public.pages.page.details_view import get_page_details_view, get_page_details_by_id_view
from wiki.api_v2.public.pages.page.import_view import page_import
from wiki.api_v2.public.pages.page.schemas import Ms365UploadSessionSchema, LinkResponse
from wiki.api_v2.public.pages.page.update_view import update_page_details
from wiki.api_v2.public.pages.request_author_role_view import request_author_role_view, grant_author_role_view
from wiki.api_v2.public.pages.revisions.router import inject_revisions_routes
from wiki.api_v2.public.pages.schemas import (
    CheckAccessResponse,
    NavigationTreeSchema,
    PageFullDetailsSchema,
    PageSchema,
    StubSchema,
    ForbiddenStubSchema,
    AclSchema,
    RequestAuthorRoleResponseSchema,
)
from wiki.api_v2.public.pages.stubs.views import stub_view, forbidden_stub_view
from wiki.api_v2.public.pages.subscribers.router import inject_subscribers_routes
from wiki.api_v2.public.pages.change_order_view import change_order_view
from wiki.api_v2.public.pages.suggest_slug_view import suggest_slug_view, SuggestSlugResponse
from wiki.api_v2.router_factory import make_public_api_router
from wiki.api_v2.schemas import DELETED_RESPONSE
from wiki.api_v2.schemas import StatusOk
from wiki.async_operations.consts import OperationCreatedSchema
from wiki.utils.features.get_features import get_wiki_features

page_router = make_public_api_router(['pages'])

page_router.get('', response=PageFullDetailsSchema, exclude_unset=True)(get_page_details_view)
page_router.post('', response=Union[PageFullDetailsSchema, Ms365UploadSessionSchema], exclude_unset=True)(
    create_page_view
)

# page actions
page_router.get('/{digit:idx}', response=PageFullDetailsSchema, exclude_unset=True, url_name='page_details_by_id')(
    get_page_details_by_id_view
)
page_router.post('/{digit:idx}', response=PageFullDetailsSchema, exclude_unset=True)(update_page_details)
page_router.delete('/{digit:idx}', response=DELETED_RESPONSE)(delete_page_view)

# all pages-wide actions
page_router.get('/{digit:idx}/backlinks', response=Collection[PageSchema])(backlinks_view)

page_router.post('/{digit:idx}/request_author_role', response=RequestAuthorRoleResponseSchema)(request_author_role_view)
page_router.post('/{digit:idx}/grant_author_role', response=RequestAuthorRoleResponseSchema)(grant_author_role_view)

page_router.post('/{digit:idx}/clone', response=OperationCreatedSchema)(clone_page_view)
page_router.post('/{digit:idx}/change_order', response=StatusOk)(change_order_view)
page_router.get('/{digit:idx}/descendants', response=Collection[PageSchema])(descendants_view)
page_router.get('/{digit:idx}/inheritable_acl', response=AclSchema)(get_inherited_acl)
page_router.get('/{digit:idx}/grids', response=Collection[PageGridsSchema])(page_grids_view)
page_router.post('/{digit:idx}/append_content', response=PageFullDetailsSchema)(append_content_view)

if get_wiki_features().ms365:  # вьюшки, работающие для ms365-страниц
    page_router.post('/{digit:idx}/ms365/ensure_access', response=StatusOk)(ensure_m365_access_view)
    page_router.post('/{digit:idx}/ms365/check_access', response=CheckAccessResponse)(check_m365_access_view)
    page_router.get('/{digit:idx}/ms365/source', response=LinkResponse)(download_ms365_source_view)

page_router.get('/attachments/download_by_url')(download_by_filename_slug_pair_view)
page_router.get('/autocomplete', response=AutoCompleteResponse)(autocomplete_view)
page_router.get('/backgrounds', response=BackgroundsResponse)(backgrounds_view)
page_router.get('/forbidden_stub', response=ForbiddenStubSchema, exclude_unset=True)(forbidden_stub_view)
page_router.get('/homepage', response=HomepageResponse)(homepage_view)
page_router.post('/import', response=PageFullDetailsSchema, exclude_unset=True)(page_import)
page_router.post('/move', response=OperationCreatedSchema)(move_view)
page_router.get('/navigation_tree', response=NavigationTreeSchema)(navigation_tree_view)
page_router.get('/stub', response=StubSchema, exclude_unset=True)(stub_view)
page_router.get('/suggest_slug', response=SuggestSlugResponse)(suggest_slug_view)

inject_comment_routes(page_router, ['pages/comments'], '/{digit:idx}/comments')
inject_attachment_routes(page_router, ['pages/attachments'], '/{digit:idx}/attachments')
inject_revisions_routes(page_router, ['pages/revisions'], '/{digit:idx}/revisions')
inject_subscribers_routes(page_router, ['pages/subscribers'], '/{digit:idx}/subscribers')
inject_access_request_routes(page_router, ['pages/access_requests'], '/{digit:idx}/access_requests')
