from datetime import datetime
from enum import Enum
from typing import Optional, List, Union, Literal, Annotated
from django.conf import settings
from ninja import Schema
from pydantic import Field, HttpUrl

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.api_v2.collections import Collection
from wiki.api_v2.public.pages.consts import SubscriptionStatus
from wiki.grids.consts import GridContentSchema
from wiki.integrations.consts import Ms365AclManagementType
from wiki.integrations.ms.consts import Ms365DocType
from wiki.intranet.models.consts import GroupType, GROUP_TYPE_CHOICES_TO_ENUM, CONNECT_GROUP_TYPE_TO_ENUM
from wiki.pages.models.consts import ActualityStatus, AclType, PageType, DB_PAGE_TYPES_MAP, ACL_TYPE_DESCRIPTION


class UserSchema(Schema):
    id: int
    identity: Optional[UserIdentity]
    username: str
    display_name: str
    is_dismissed: bool


class BreadcrumbSchema(Schema):
    id: int = None
    title: str
    slug: str
    page_exists: bool


class MovedOutPage(Schema):
    accessible: bool = Field(..., description='Доступна ли страница пользователю')
    slug: str
    title: str = Field(..., description='Если страница не доступна, `title == slug`')

    @classmethod
    def serialize(cls, page, accessible: bool) -> 'MovedOutPage':
        return cls(accessible=accessible, slug=page.slug, title=page.title if accessible else page.slug)


class IntranetGroupMetadata(Schema):
    url: Optional[str]
    externals_count: Optional[int]


class IamGroupMetadata(Schema):
    dir_id: str


def _get_formatted_url(group):
    """
    Сгенерировать урл для группы.
    Для департаментных групп — на Стафф.
    Для сервисных групп — на Планер.
    Для викигрупп — None, потому что мы не поощряем их использование.
    """
    DEPARTMENT_URL_TPL = settings.STAFF_URL + 'departments/{url}/'
    SERVICE_URL_TPL = settings.PLANNER_URL + 'services/{service_id}/'
    if group.is_department:
        return DEPARTMENT_URL_TPL.format(url=group.url)
    elif group.is_service:
        return SERVICE_URL_TPL.format(service_id=group.service_id)


class GroupSchema(Schema):
    id: str = Field(description='Идентификатор группы. Для внешнего инстанса - dir_id, для внутреннего - staff_id')
    name: str
    type: GroupType
    metadata: Union[IntranetGroupMetadata, IamGroupMetadata] = None

    @classmethod
    def serialize(cls, group) -> 'GroupSchema':
        if settings.IS_INTRANET:
            metadata = IntranetGroupMetadata(url=_get_formatted_url(group), externals_count=group.externals_count)
            q = cls(
                id=group.get_public_group_id(),
                name=group.name,
                type=GROUP_TYPE_CHOICES_TO_ENUM[group.type],
            )
            # Union is type coerced without notification otherwise during construction
            q.metadata = metadata
            return q
        elif settings.IS_BUSINESS:
            metadata = IamGroupMetadata(
                dir_id=group.dir_id,
            )
            q = cls(
                id=group.get_public_group_id(),
                name=group.title,
                type=CONNECT_GROUP_TYPE_TO_ENUM[group.group_type],
            )
            # Union is type coerced without notification otherwise during construction
            q.metadata = metadata
            return q
        else:
            raise ValueError('What are you even?')


class GroupWithMembersSchema(GroupSchema):
    members: Optional[Collection[UserSchema]]


class CheckOrgMembershipSchema(Schema):
    is_member: bool


class OfficialitySchema(Schema):
    is_official: bool
    responsible_groups: Optional[List[GroupSchema]]
    responsible_persons: Optional[List[UserSchema]]


class PageAttributesSchema(Schema):
    created_at: datetime
    modified_at: datetime
    lang: str
    is_readonly: bool
    comments_count: int
    comments_enabled: bool
    keywords: Optional[List[str]]


class PageSchema(Schema):
    id: int
    slug: str

    @classmethod
    def serialize(cls, page) -> 'PageSchema':
        return cls(id=page.id, slug=page.supertag)


class AclPreviewSchema(Schema):
    type: AclType = Field(description=ACL_TYPE_DESCRIPTION)
    is_readonly: bool
    inherits_from: Optional[str]


class AclSchema(Schema):
    acl_type: AclType = Field(description=ACL_TYPE_DESCRIPTION)
    inherits_from: Optional[PageSchema]

    break_inheritance: bool
    is_readonly: bool

    users: Optional[List[UserSchema]]
    groups: Optional[List[GroupSchema]]


class ClusterSchema(Schema):
    slug: str


class AuthorSchema(Schema):
    owner: Optional[UserSchema]
    last_author: Optional[UserSchema]
    all: List[UserSchema]


class AuthorUpdateSchema(Schema):
    owner: Optional[UserIdentity]
    all: List[UserIdentity]


class BackgroundImageSchema(Schema):
    id: int
    url: HttpUrl
    preview: HttpUrl
    type: Literal['image']


class BackgroundColorSchema(Schema):
    id: int
    color: str
    type: Literal['color']


class BackgroundGradientSchema(Schema):
    id: int
    gradient: str
    type: Literal['gradient']


class BackgroundSchema(Schema):
    __root__: Annotated[
        Union[BackgroundImageSchema, BackgroundColorSchema, BackgroundGradientSchema],
        Field(discriminator='type'),
    ]


class CommentSchema(Schema):
    id: int
    body: str
    parent_id: Optional[int]
    author: UserSchema
    created_at: datetime
    is_deleted: bool


class CreateCommentSchema(Schema):
    body: str
    parent_id: Optional[int]


class EditCommentSchema(Schema):
    body: str


class CloudPageEmbeddingSchema(Schema):
    iframe_src: str = Field(description='SRC для iframe')
    edit_src: str = Field(description='URL для открытия в новом окне для редактирования документа в MS365')


class CloudPageContentSchema(Schema):
    embed: CloudPageEmbeddingSchema
    acl_management: Ms365AclManagementType
    type: Ms365DocType
    filename: str


class PageDetailsSchema(Schema):
    id: int
    slug: str
    title: str
    page_type: PageType

    @classmethod
    def serialize(cls, page, revision=None):
        from wiki.users.user_data_repository import USER_DATA_REPOSITORY

        active_revision = None
        if revision:
            active_revision = RevisionSchema(
                id=revision.id,
                author=USER_DATA_REPOSITORY.orm_to_user_schema(revision.author),
                created_at=revision.created_at,
            )

        return cls(
            id=page.id,
            slug=page.supertag,
            title=page.title if revision is None else revision.title,
            active_revision=active_revision,
            page_type=DB_PAGE_TYPES_MAP[page.page_type],
        )


class RedirectSchema(Schema):
    page_id: Optional[int] = Field(description='ID страницы на которую эта страница перенаправляет')
    redirect_target: Optional[PageDetailsSchema] = Field(
        description='Если есть цепочка редиректов A->B->C->D, ' 'для A,B и С, это будет страница D'
    )


class ActualitySchema(Schema):
    status: ActualityStatus
    marked_at: Optional[datetime]

    user: Optional[UserSchema]
    comment: Optional[str]

    external_links: Optional[List[str]]
    actual_pages: Optional[List[PageDetailsSchema]]


class PersonalContextSchema(Schema):
    is_bookmarked: bool = Field(description='Есть ли страница у пользователя в избранном')
    subscription: Optional[SubscriptionStatus] = Field(
        description='Подписан на кластер или на страницу. `null` если не подписан'
    )


class PageDetailsBookmarkSchema(Schema):
    id: Optional[int] = Field(description='ID закладки, если страница в избранном; `null` в противном случае')


class PageDetailsSubscriptionSchema(Schema):
    status: Optional[SubscriptionStatus] = Field(
        description='Подписан на кластер или на страницу. `null` если не подписан'
    )


def _mk_hint(fieldname, fulldesc=''):
    if fulldesc != '':
        fulldesc = '\n\n' + fulldesc

    return Field(
        description=f'**По-умолчанию поле не отдается**\n\n'
        f'Укажите `{fieldname}` в списке `?fields={fieldname}, ...` для получения в ответе.{fulldesc}'
    )


CONTENT_DESC = """Разный для разных типов страницы:
- `page`, `wysiwyg` - `str`
- `cloud_page` - `CloudPageContentSchema`
- `grid` - `GridContentSchema`"""


class UserPermission(str, Enum):
    CAN_CREATE_PAGE = 'create_page'
    CAN_DELETE = 'delete'
    CAN_EDIT = 'edit'
    CAN_VIEW = 'view'
    CAN_COMMENT = 'comment'
    CAN_CHANGE_AUTHORS = 'change_authors'
    CAN_CHANGE_ACL = 'change_acl'


class RevisionSchema(Schema):
    id: int
    author: UserSchema
    created_at: datetime


class PageFullDetailsSchema(PageDetailsSchema):
    cluster: Optional[ClusterSchema] = _mk_hint('cluster')
    redirect: Optional[RedirectSchema] = _mk_hint('redirect', 'Если редиректа нет, придет `null`')
    authors: Optional[AuthorSchema] = _mk_hint('authors')
    access: Optional[AclPreviewSchema] = _mk_hint('access')
    acl: Optional[AclSchema] = _mk_hint('acl')
    breadcrumbs: Optional[List[BreadcrumbSchema]] = _mk_hint('breadcrumbs')
    attributes: Optional[PageAttributesSchema] = _mk_hint('attributes')
    content: Optional[Union[CloudPageContentSchema, GridContentSchema, str]] = _mk_hint('content', CONTENT_DESC)
    officiality: Optional[OfficialitySchema] = _mk_hint('officiality')
    actuality: Optional[ActualitySchema] = _mk_hint('actuality')
    bookmark: Optional[PageDetailsBookmarkSchema] = _mk_hint('bookmark', 'Если страница не в избранном, придет `null`')
    subscription: Optional[PageDetailsSubscriptionSchema] = _mk_hint('subscription', 'Если нет подписки, придет `null`')
    user_permissions: Optional[List[UserPermission]] = _mk_hint('user_permissions')
    background: Optional[BackgroundSchema] = _mk_hint('background', 'Если нет подложки, придет `null`')

    last_revision_id: Optional[int] = _mk_hint('last_revision_id')
    active_revision: Optional[RevisionSchema] = Field(
        description='Если при запросе страницы указать прошлую ревизию, поле будет заполнено информацией о ней'
    )


class StubSchema(Schema):
    cluster: Optional[ClusterSchema] = _mk_hint('cluster')
    breadcrumbs: Optional[List[BreadcrumbSchema]] = _mk_hint('breadcrumbs')
    user_permissions: Optional[List[UserPermission]] = _mk_hint('user_permissions')
    moved_out_pages: Optional[List[MovedOutPage]] = _mk_hint('moved_out_pages')
    slug: str
    title: str

    @classmethod
    def serialize(cls, slug):
        return cls(slug=slug, title=slug.rsplit('/', 1)[-1])


class NavigationTreeNode(Schema):
    page_id: Optional[int]
    slug: str
    title: str
    page_type: Optional[PageType] = Field(
        description='Для страниц, которых еще не существует или для них нет доступа будет null.'
    )
    is_missing: bool
    children_count: int


class NavigationTreeSchema(Schema):
    node: NavigationTreeNode
    children: List[NavigationTreeNode]
    has_next: bool


class DocumentAccessLevel(str, Enum):
    NO_ACCESS = 'denied'
    READ_WRITE = 'read_write'
    READ = 'read'


class CheckAccessResponse(Schema):
    access_level: DocumentAccessLevel


class AccessRequestSchema(Schema):
    requested_at: datetime
    reason: str


class ForbiddenStubSchema(Schema):
    cluster: Optional[ClusterSchema] = _mk_hint('cluster')
    breadcrumbs: Optional[List[BreadcrumbSchema]] = _mk_hint('breadcrumbs')
    authors: Optional[AuthorSchema] = _mk_hint('authors')
    pending_access_request: Optional[AccessRequestSchema] = _mk_hint('pending_access_request')
    id: int
    slug: str
    title: str

    @classmethod
    def serialize(cls, page):
        return cls(id=page.id, slug=page.slug, title=page.slug.rsplit('/', 1)[-1])


class RequestAuthorRoleSchema(Schema):
    reason: Optional[str]


class GrantAuthorRoleSchema(Schema):
    user: UserIdentity


class RequestAuthorRoleResult(str, Enum):
    ROLE_GRANTED = 'role_granted'
    REQUEST_SENT = 'success'
    REQUEST_ALREADY_SENT = 'already_sent'
    ALREADY_AUTHOR = 'already_author'


class RequestAuthorRoleDetails(str, Enum):
    ADMIN = 'admin'
    EVERYONE_DISMISSED = 'everyone_dismissed'
    VIA_INHERITANCE = 'via_inheritance'


class RequestAuthorRoleResponseSchema(Schema):
    result: RequestAuthorRoleResult
    details: Optional[RequestAuthorRoleDetails]
