from typing import List

from wiki.api_v2.collections import CompositeSerializer
from wiki.api_v2.public.pages.page.getters import (
    get_cluster_schema_from_slug,
    get_moved_out_pages_for_slug,
    get_authors_schema,
    get_pending_access_request,
    get_cluster_schema_from_page,
)
from wiki.api_v2.public.pages.page.permissions import get_user_permissions_for_slug
from wiki.api_v2.public.pages.schemas import (
    StubSchema,
    ForbiddenStubSchema,
)
from wiki.api_v2.schemas import Slug
from wiki.pages.logic.breadcrumbs import get_breadcrumbs_from_slug, get_breadcrumbs_from_page
from wiki.pages.models import Page


class StubCompositeSerializer(CompositeSerializer[Slug, StubSchema]):
    FIELDS = {
        'cluster': (get_cluster_schema_from_slug, []),
        'breadcrumbs': (get_breadcrumbs_from_slug, ['org']),
        'user_permissions': (get_user_permissions_for_slug, ['org', 'user']),
        'moved_out_pages': (get_moved_out_pages_for_slug, ['org', 'user']),
    }

    @classmethod
    def serialize(cls, stub_slug: Slug, fields: List[str], request, organization) -> StubSchema:
        base = StubSchema.serialize(stub_slug)
        return cls._add_fields(base, stub_slug, fields, {'request': request, 'user': request.user, 'org': organization})


class ForbiddenStubCompositeSerializer(CompositeSerializer[Page, ForbiddenStubSchema]):
    FIELDS = {
        'cluster': (get_cluster_schema_from_page, []),
        'breadcrumbs': (get_breadcrumbs_from_page, []),
        'authors': (get_authors_schema, []),
        'pending_access_request': (get_pending_access_request, ['user']),
    }

    @classmethod
    def serialize(cls, page: Page, fields: List[str], request, organization) -> ForbiddenStubSchema:
        base = ForbiddenStubSchema.serialize(page)
        return cls._add_fields(base, page, fields, {'user': request.user, 'org': organization})
