from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import BadRequest, IsStubPage
from wiki.api_v2.public.pages.schemas import StubSchema, ForbiddenStubSchema
from wiki.api_v2.public.pages.stubs.serializer import StubCompositeSerializer, ForbiddenStubCompositeSerializer
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.api_v2.schemas import Slug
from wiki.pages.logic.etalon import get_etalon_page
from wiki.personalisation.user_cluster import expand_user_cluster_alias
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.supertag import normalize_supertag
from django.conf import settings


class PageExisits(BadRequest):
    error_code = 'PAGE_EXISTS'
    debug_message = 'Page exists, can not gather placeholder info'


@di
@legacy_org_ctx
def stub_view(request, organization: BaseOrganization, slug: Slug, fields: str = None) -> StubSchema:
    """
    Получить данные для страницы-заглушки по определенному адресу:
    - к какому кластеру относится
    - какие хлебные крошки
    - какие страницы раньше тут были
    - может ли пользователь создать страницу по этому урлу
    """
    fields = StubCompositeSerializer.extract_fields(fields)
    slug = expand_user_cluster_alias(slug, request.user)
    slug = normalize_supertag(slug)

    log_slug(request, slug=slug)

    if slug in settings.COPY_ON_WRITE_TAGS:
        raise PageExisits()

    if organization.get_active_pages().filter(supertag=slug).exists():
        raise PageExisits()

    return StubCompositeSerializer.serialize(slug, fields, request, organization)


@di
@legacy_org_ctx
def forbidden_stub_view(request, organization: BaseOrganization, slug: Slug, fields: str = None) -> ForbiddenStubSchema:
    """ """
    fields = ForbiddenStubCompositeSerializer.extract_fields(fields)
    slug = expand_user_cluster_alias(slug, request.user)
    slug = normalize_supertag(slug)

    log_slug(request, slug=slug)

    try:
        page = get_page_or_404(organization, supertag=slug)
    except IsStubPage:
        page = get_etalon_page(organization, slug=slug)

    return ForbiddenStubCompositeSerializer.serialize(page, fields, request, organization)
