from itertools import chain
from typing import Optional, Union

from django.contrib.auth import get_user_model
from django.db.models import Q
from django.shortcuts import get_object_or_404
from ninja import Path
from ninja import Query
from wiki.acl.check_access import assert_has_access, _check_access
from wiki.acl.consts import Action
from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.api_v2.collections import Collection, PaginationQuery, CollectionFactory
from wiki.api_v2.di import di, legacy_org_ctx, log_slug
from wiki.api_v2.exceptions import BadRequest, IsStubPage
from wiki.api_v2.public.pages.subscribers.schemas import (
    AddSubscribersResponse,
    AddSubscribersSchema,
    SubscriberSchema,
)
from wiki.api_v2.public.utils.get_object import get_page_or_404
from wiki.api_v2.schemas import DELETED
from wiki.pages.logic import subscription as legacy_subscription
from wiki.pages.logic.etalon import create_etalon_page, get_etalon_page
from wiki.pages.models import PageWatch
from wiki.subscriptions.exceptions import SubscribeWithoutAccess
from wiki.subscriptions.logic import WATCHES_ID_INCREMENT, get_page_subscriptions, create_subscriptions, get_parents
from wiki.subscriptions.models import Subscription, SubscriptionType
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.users.dao import get_users_by_identity, UserNotFound
from wiki.users.logic.settings import uses_new_subscriptions
from wiki.users.user_data_repository import USER_DATA_REPOSITORY

User = get_user_model()


def serialize(subscription: Union[SubscriberSchema, Subscription]) -> SubscriberSchema:
    if isinstance(subscription, SubscriberSchema):
        return subscription

    return SubscriberSchema(
        id=subscription.id,
        is_cluster=subscription.is_cluster,
        user=USER_DATA_REPOSITORY.orm_to_user_schema(subscription.user),
    )


@di
@legacy_org_ctx
def get_subscribers_view(
    request,
    organization: BaseOrganization,
    idx: int = Path(..., description='Page id'),
    q: Optional[str] = Query(None, description='Filter by the beginning of the full name or login. Case-insensitive'),
    pagination: PaginationQuery = Query(...),
) -> Collection[SubscriberSchema]:
    """Get subscribers of page (idx) and filter by name or username using a q (query)"""

    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    subscriptions = get_page_subscriptions(page, q)
    # Исключаем пользователей, которые не используют новую схему подписок
    subscriptions = [s for s in subscriptions if uses_new_subscriptions(s.user)]

    # Добавляем пользователей, которые используют старую схему
    watches = legacy_subscription.get_page_watches(page)
    subscribers = []
    for watch in watches:
        user = organization.get_users().filter(username=watch.user).first()
        if user and not uses_new_subscriptions(user):
            subscribers.append(
                SubscriberSchema(
                    id=watch.id + WATCHES_ID_INCREMENT,
                    is_cluster=watch.is_cluster,
                    user=USER_DATA_REPOSITORY.orm_to_user_schema(user),
                )
            )

    return CollectionFactory.pagination_build(
        qs=(subscriptions + subscribers),
        serializer=serialize,
        pagination=pagination,
    )


@di
@legacy_org_ctx
def add_subscribers_view(
    request,
    organization: BaseOrganization,
    data: AddSubscribersSchema,
    idx: int = Path(..., description='Page id'),
) -> AddSubscribersResponse:
    """Add subscribers to page (idx). The added users also must have read access"""

    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = create_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    try:
        users = set(get_users_by_identity(data.users, organization=organization, apiv1_exceptions=False))
    except UserNotFound as u:
        raise BadRequest(
            'Some of requested users not found',
            details={
                'missing': [m.dict(exclude_unset=True) for m in u.missing_identities],
            },
        )

    legacy_subscriptions_users = {user for user in users if not uses_new_subscriptions(user)}
    users = set(users) - legacy_subscriptions_users

    for user in legacy_subscriptions_users:  # для `users` с новой моделью проверяем доступы ниже, при создании подписок
        if not _check_access(user, organization=organization, slug='', page=page, op=Action.READ):
            raise SubscribeWithoutAccess(details={'user_identity': UserIdentity.from_user(user).dict()})

    subscriptions = []
    if users:
        type_ = SubscriptionType.OTHER
        subscriptions = create_subscriptions(users=users, page=page, type_=type_, is_cluster=data.is_cluster)

    subscribers = []
    for user in legacy_subscriptions_users:
        if data.is_cluster:
            # При подписке на кластер в старых подписках происходит подписка на каждую
            # страницу в дереве, не вариант возвращать их все, вернем только одну подписку
            # на корневую страницу
            legacy_subscription.subscribe_user_to_cluster(user, page, request.user, '')
        else:
            legacy_subscription.create_watch(page, user, False)

        if watch := PageWatch.objects.filter(page=page, user=user.username).first():
            subscribers.append(
                SubscriberSchema(
                    id=watch.id + WATCHES_ID_INCREMENT,
                    is_cluster=watch.is_cluster,
                    user=USER_DATA_REPOSITORY.orm_to_user_schema(user),
                )
            )

    return AddSubscribersResponse(
        results=[serialize(subscription) for subscription in chain(subscriptions, subscribers)]
    )


@di
@legacy_org_ctx
def delete_subscriber_view(
    request,
    organization: BaseOrganization,
    idx: int = Path(..., description='Page id'),
    subscription_id: int = Path(...),
):
    """Delete subscriber from page (idx) by `subscription_id`"""

    try:
        page = get_page_or_404(organization, pk=idx)
    except IsStubPage:
        page = get_etalon_page(organization, idx=idx)

    log_slug(request, slug=page.slug)
    assert_has_access(request.user, organization, page, Action.READ)

    if subscription_id > WATCHES_ID_INCREMENT:
        # Старая подписка
        subscription_id -= WATCHES_ID_INCREMENT
        watch = get_object_or_404(PageWatch, pk=subscription_id, page=page)
        watch.delete()
    else:
        query = Q(page=page) | Q(page__supertag__in=get_parents(slug=page.slug), is_cluster=True)
        subscription = get_object_or_404(Subscription, query, pk=subscription_id)
        subscription.delete()

    return DELETED
