from datetime import datetime
from typing import List, Set
from uuid import uuid4

from ninja import Schema
from wiki.api_v2.di import di, log_slug
from wiki.api_v2.exceptions import ValidationError
from wiki.api_v2.public.pages.exceptions import SlugTooLong, SlugEmpty
from wiki.personalisation.user_cluster import expand_user_cluster_alias
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.supertag import normalize_supertag

from pydantic import constr


LIMIT_TITLE_LENGTH = 50  # число выбрано эмпирически


class SuggestSlugResponse(Schema):
    slug: constr(min_length=1, max_length=255)
    occupied: bool
    suggest: List[str]


@di
def suggest_slug_view(
    request, organization, current_slug: str = '', title: str = '', slug: str = ''
) -> SuggestSlugResponse:
    """
    Выдает информацию о том, доступна страница или нет. Работаем в двух режимах:

    - current_slug: текущий раздел, куда пользователь собирается создать страницу, обязательный параметр (может быть пустым)
    - title: желаемый заголовок страницы (кириллица)

    или

    - slug: желаемый слаг
    """

    slug_provided_mode = bool(slug)
    normalize_title_mode = bool(title)

    if not normalize_title_mode and not slug_provided_mode:
        raise ValidationError('Provide either current_slug+title or slug')
    if normalize_title_mode and slug_provided_mode:
        raise ValidationError('Provide either current_slug+title or slug, not both!')

    if normalize_title_mode:
        current_slug = normalize_supertag(expand_user_cluster_alias(current_slug, user=request.user))
        normalized_title = normalize_supertag(title.replace('/', ''))[:LIMIT_TITLE_LENGTH]
        if not normalized_title:
            raise SlugEmpty()

        if current_slug:
            slug: str = current_slug + '/' + normalized_title
        else:
            slug: str = normalized_title
    else:
        slug = normalize_supertag(expand_user_cluster_alias(slug, user=request.user))

    log_slug(request, slug=slug)

    if len(slug) == 0:
        raise SlugEmpty()
    if len(slug) > 255:
        raise SlugTooLong(details={'slug': slug})

    occupied = False
    suggest = []

    if is_slug_occupied(organization, slug):
        occupied = True
        suggest = suggest_unoccupied_slugs(organization, slug)

    return SuggestSlugResponse(slug=slug, occupied=occupied, suggest=suggest)


def find_occupied_slugs(org: BaseOrganization, slugs: Set[str]) -> Set[str]:
    return set(org.get_pages().filter(supertag__in=slugs).values_list('supertag', flat=True))


def is_slug_occupied(org: BaseOrganization, slug: str) -> Set[str]:
    return org.get_pages().filter(supertag=slug).exists()


def generate_similar_slug(slug: str) -> List[str]:
    today = datetime.today()
    arr = []

    arr.append(f'{slug}-1'[:255])
    arr.append(f'{slug}-2'[:255])
    arr.append(f'{slug}-{today.day}{today.month}'[:255])
    arr.append(f'{slug}-{today.day}-{today.month}'[:255])
    arr.append(f'{slug}-{str(uuid4().hex)[:6]}'[:255])

    return arr


def suggest_unoccupied_slugs(org: BaseOrganization, slug: str) -> List[str]:
    generated = set(generate_similar_slug(slug))
    occupied_slugs = find_occupied_slugs(org, generated)
    return list(generated - occupied_slugs)
