from datetime import datetime
from typing import Optional

from ninja import Schema
from pydantic import UUID4, constr

from wiki.api_v2.public.pages.schemas import UserSchema
from wiki.uploads.consts import UploadSessionTargetType, UploadSessionStatusType
from wiki.uploads.models.session import UploadSession
from wiki.users.user_data_repository import USER_DATA_REPOSITORY

FILE_NAME_REGEX = r'^[\w\-.\s()]+$'


class CreateUploadSessionSchema(Schema):
    target: UploadSessionTargetType
    file_name: constr(regex=FILE_NAME_REGEX)
    file_size: int


class UploadSessionSchema(Schema):
    session_id: UUID4
    file_name: str
    file_size: int
    status: UploadSessionStatusType
    target: UploadSessionTargetType
    user: UserSchema
    created_at: datetime
    finished_at: Optional[datetime]

    @classmethod
    def serialize(cls, upload_session: UploadSession) -> 'UploadSessionSchema':
        return cls(
            session_id=str(upload_session.session_id),
            file_name=upload_session.file_name,
            file_size=upload_session.file_size,
            status=upload_session.status,
            target=upload_session.target,
            user=USER_DATA_REPOSITORY.orm_to_user_schema(upload_session.user),
            created_at=upload_session.created_at,
            finished_at=upload_session.finished_at,
        )
