from django.db import transaction
from django.http import HttpRequest
from django.shortcuts import get_object_or_404
from pydantic import UUID4

from wiki.api_v2.di import di
from wiki.api_v2.public.upload_sessions.exceptions import (
    MalformedUploadSessionRequest,
)
from wiki.api_v2.public.upload_sessions.schemas import (
    UploadSessionSchema,
    CreateUploadSessionSchema,
)
from wiki.api_v2.schemas import StatusOk
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.uploads.consts import UploadSessionStatusType, Chunk, PartNumber
from wiki.uploads.logic import (
    check_limit_sessions,
    check_limit_size,
    create_upload_session,
    check_session_non_exists,
)
from wiki.uploads.models.session import UploadSession
from wiki.uploads.s3_client import S3_CLIENT
from wiki.utils import timezone
from wiki.utils.django_redlock.redlock import RedisLock

CONTENT_TYPE_OCTET_STREAM = 'application/octet-stream'


@di
def get_upload_session_view(request, organization: BaseOrganization, session_id: UUID4) -> UploadSessionSchema:
    # Можно добавить проверку на то, что пользователь - это админ, тогда выводим сессию для него
    upload_session = get_object_or_404(
        UploadSession.objects, session_id=session_id, user=request.user, org=organization.as_django_model()
    )

    return UploadSessionSchema.serialize(upload_session)


@di
def create_upload_session_view(
    request, organization: BaseOrganization, data: CreateUploadSessionSchema
) -> UploadSessionSchema:
    """
    Процедура загрузки:

    1. Делаем заявку на то чтобы загрузить файл POST на `/api/v2/public/upload_sessions`
    2. Получаем `session_id` и делаем серию `PUT` на
    `/api/v2/public/upload_sessions/{session_id}/upload_part?part_number=1..`
      - `Content-Type: application/octet-stream`
      -  Файл разделяется на части размером от 5мб до 16мб (кроме последней части). Части стримим в body
      -  Грузим части на `/api/v2/public/upload_sessions/{session_id}/upload_part?part_number=1` увеличивая part_number
      -  Части можно грузить параллельно и перезагружать при ошибке
    3. Когда все загружено, финализируем POST `/api/v2/public/upload_sessions/{session_id}/finish`
    4. Если пользователь отменил загрузку - POST `/api/v2/public/upload_sessions/{session_id}/abort`
    """
    with RedisLock(
        name=f'CreateUploadSession: {request.user}, {organization}, {data.file_name}, '
        f'{data.file_size}, {data.target}'
    ):
        check_session_non_exists(request.user, organization, data)
        check_limit_sessions(request.user, organization)
        check_limit_size(data)

        upload_session = create_upload_session(request.user, organization, data)

    return UploadSessionSchema.serialize(upload_session)


@di
def upload_part_view(
    request: HttpRequest, organization: BaseOrganization, session_id: UUID4, part_number: PartNumber
) -> UploadSessionSchema:
    """
    Все куски кроме последнего должны быть >=5mb
    """
    if request.content_type != CONTENT_TYPE_OCTET_STREAM:
        raise MalformedUploadSessionRequest(debug_message='Only application/octet-stream is accepted')

    size = int(request.META.get('CONTENT_LENGTH') or 0)

    if size > 16 * 1024 * 1024:
        raise MalformedUploadSessionRequest(debug_message='Please, keep chunks less than 16Mb')

    """
    request._stream is set by a corresponding request subclass (e.g. WSGIRequest).
    hopefully, at this point noone read _stream before
    """
    if request._read_started:
        raise ValueError('Request stream was touched before! (through middleware or something like that)')

    body = request.read()

    upload_session: UploadSession = get_object_or_404(
        UploadSession.objects, session_id=session_id, user=request.user, org=organization.as_django_model()
    )

    if upload_session.status != UploadSessionStatusType.IN_PROGRESS:
        raise MalformedUploadSessionRequest(debug_message=f'Session already in status {upload_session.status}')

    if upload_session.uploaded_size > upload_session.file_size:
        raise MalformedUploadSessionRequest(debug_message='Wont accept more data into this session')

    etag = S3_CLIENT.upload_part(
        body, key=upload_session.storage_key, part_number=part_number, upload_id=upload_session.upload_id
    )

    chunk = Chunk(part_number=part_number, size=size, etag=etag)

    with transaction.atomic():
        upload_session = UploadSession.objects.select_for_update().get(session_id=upload_session.session_id)
        upload_session.add_chunk(chunk)
        upload_session.save()

    return UploadSessionSchema.serialize(upload_session)


@di
def abort_multipart_upload_view(request, organization: BaseOrganization, session_id: UUID4) -> UploadSessionSchema:
    upload_session: UploadSession = get_object_or_404(
        UploadSession.objects, session_id=session_id, user=request.user, org=organization.as_django_model()
    )

    if upload_session.status != UploadSessionStatusType.IN_PROGRESS:
        raise MalformedUploadSessionRequest(debug_message=f'Session already in status {upload_session.status}')

    S3_CLIENT.abort_multipart_upload(upload_session.storage_key, upload_id=upload_session.upload_id)

    with transaction.atomic():
        upload_session = UploadSession.objects.select_for_update().get(session_id=upload_session.session_id)
        upload_session.status = UploadSessionStatusType.ABORTED
        upload_session.save()

    return UploadSessionSchema.serialize(upload_session)


@di
def abort_all_view(request, organization: BaseOrganization) -> StatusOk:
    """
    Отменить все активные закачки пользователя чтобы освободить квоту
    """
    active_sessions = UploadSession.objects.filter(
        user=request.user, org=organization.as_django_model(), status=UploadSessionStatusType.IN_PROGRESS
    )
    for upload_session in active_sessions:
        with transaction.atomic():
            S3_CLIENT.abort_multipart_upload(upload_session.storage_key, upload_id=upload_session.upload_id)
            upload_session.status = UploadSessionStatusType.ABORTED
            upload_session.save()

    return StatusOk()


@di
def complete_multipart_upload_view(request, organization: BaseOrganization, session_id: UUID4) -> UploadSessionSchema:
    """
    Все куски кроме последнего должны быть >=5mb
    """
    upload_session: UploadSession = get_object_or_404(
        UploadSession.objects, session_id=session_id, user=request.user, org=organization.as_django_model()
    )

    if upload_session.status != UploadSessionStatusType.IN_PROGRESS:
        raise MalformedUploadSessionRequest(debug_message=f'Session already in status {upload_session.status}')

    if upload_session.uploaded_size != upload_session.file_size:
        raise MalformedUploadSessionRequest(
            debug_message=f'Incomplete upload: declared - {upload_session.file_size}, '
            f'actual - {upload_session.uploaded_size}'
        )

    S3_CLIENT.complete_multipart_upload(
        upload_session.storage_key, upload_id=upload_session.upload_id, chunks=upload_session.get_chunks()
    )

    upload_session.status = UploadSessionStatusType.FINISHED
    upload_session.finished_at = timezone.now()
    upload_session.save()

    return UploadSessionSchema.serialize(upload_session)
