from typing import Optional

from ninja import Schema
from ninja.schema import ResolverMetaclass
from pydantic import PydanticValueError, constr

from wiki.utils.supertag import normalize_supertag

NotEmptyString = constr(min_length=1, max_length=255)
NotEmptyStringTrimmed = constr(min_length=1, max_length=255, strip_whitespace=True)


class Empty(Schema):
    pass


class StatusOk(Schema):
    status: str = 'ok'


class SlugError(PydanticValueError):
    code = 'bad_slug'
    msg_template = 'page slug {normalized_slug} must be between 1 and 255 chars'


class Slug(str):
    """
    Авто-нормализует пользовательский ввод;
    must-use во всех полях где slug приходит из пользовтельского ввода
    """

    @classmethod
    def __get_validators__(cls):
        yield cls.validate

    @classmethod
    def validate(cls, v):
        v = normalize_supertag(v)
        if len(v) == 0:
            raise SlugError(normalized_slug='< empty >')
        if len(v) > 255:
            raise SlugError(normalized_slug=v[:20] + '...')
        return v


class SlugOrEmpty(Slug):
    @classmethod
    def validate(cls, v):
        v = normalize_supertag(v)
        if v == '':
            return v
        return super().validate(v)


DELETED = 204, ''
DELETED_RESPONSE = {204: Empty}

Missing = object()


class AllOptional(ResolverMetaclass):
    def __new__(self, name, bases, namespaces, **kwargs):
        annotations = namespaces.get('__annotations__', {})
        for base in bases:
            annotations = {**annotations, **base.__annotations__}
        for field in annotations:
            if not field.startswith('__'):
                annotations[field] = Optional[annotations[field]]
        namespaces['__annotations__'] = annotations
        return super().__new__(self, name, bases, namespaces, **kwargs)
