import inspect

from importlib import import_module
from django.template.loader import render_to_string

from wiki.api_v2.exceptions import NinjaApiError


def collect_api_errors():
    modules = [
        'wiki.api_v2.exceptions',
        'wiki.api_v2.public.pages.comments.exceptions',
        'wiki.api_v2.public.pages.subscribers.exceptions',
        'wiki.api_v2.public.pages.exceptions',
        'wiki.api_v2.public.me.exceptions',
        'wiki.api_v2.public.me.favorites.exceptions',
        'wiki.api_v2.public.upload_sessions.exceptions',
    ]
    errors = []
    base_class = NinjaApiError
    for module in modules:
        module = import_module(module)
        for name, obj in inspect.getmembers(module):
            if inspect.isclass(obj) and issubclass(obj, base_class):
                errors.append({
                    'name': obj.__name__,
                    'status_code': getattr(obj, 'status_code'),
                    'error_code': getattr(obj, 'error_code'),
                })

    return errors


def make_api_description():
    errors = collect_api_errors()
    errors_desc = []

    template = """
- **{name}**
  - status_code: {status_code}
  - error_code: {error_code}
"""

    for error in sorted(errors, key=lambda x: x['status_code']):
        errors_desc.append(
            template.format(name=error['name'], status_code=error['status_code'], error_code=error['error_code'])
        )

    errors_desc = '\n'.join(errors_desc)

    description = render_to_string('api_description.txt', {'errors': errors_desc})
    return description
