from enum import Enum
from typing import Dict, TypeVar, Generic, Optional
from pydantic import BaseModel, Field
from pydantic.generics import GenericModel
from ninja import Schema


class OperationType(str, Enum):
    TEST = 'test'
    EXPORT = 'export'
    MOVE_CLUSTER = 'move'
    CLONE_PAGE = 'clone'


class Progress(BaseModel):
    percentage: float
    details: str


class Status(str, Enum):
    SCHEDULED = 'scheduled'
    IN_PROGRESS = 'in_progress'
    SUCCESS = 'success'
    FAILED = 'failed'


class OperationIdentity(BaseModel):
    type: OperationType
    id: str

    def build_status_url(self):
        return f'/api/v2/public/operations/{self.type.value}/{self.id}'


T = TypeVar('T')


class OperationCreatedSchema(Schema):
    operation: OperationIdentity
    dry_run: bool = False
    status_url: str = Field('URL по которому вернется прогресс операции')


class OperationOwner(Schema):
    """
    Оба ID - в БД Вики.
    """

    org_inner_id: Optional[int]
    user_id: int


class OperationProgress(BaseModel):
    id: OperationIdentity
    owner: OperationOwner
    status: Status
    progress: Progress
    result: Dict


class AsyncOperationStatusSchema(GenericModel, Generic[T]):
    status: Status
    progress: Optional[Progress]
    result: Optional[T]


# /pages/1234/move -> {guid}
# /operations/move/{ABCD-EFGH-QURST}
