import logging
from typing import TypeVar, Generic, Type

from pydantic import BaseModel

from wiki.api_core.errors.rest_api_error import RestApiError
from wiki.async_operations.consts import OperationIdentity, OperationOwner
from wiki.async_operations.exceptions import AlreadyRunning
from wiki.async_operations.progress_storage import ProgressStorage
from wiki.async_process.backend.consts import AsyncTaskType

T = TypeVar('T', bound=BaseModel)
T2 = TypeVar('T2', bound=BaseModel)

logger = logging.getLogger(__name__)


class BaseOperation(Generic[T, T2]):
    TASK_TYPE: AsyncTaskType

    arg_class: Type[T]
    result_class: Type[T2]

    args: T

    def __init__(self, args: T, owner: OperationOwner, task_id: str = None):

        self.args = self.cleanup_args(args)
        self.owner = owner

        self._task_identity = None

        if task_id:
            self._task_identity = OperationIdentity(id=task_id, type=self.TASK_TYPE)

    def cleanup_args(self, task_args: T) -> T:
        return task_args

    def get_task_identity(self) -> OperationIdentity:
        if self._task_identity is None:
            self._task_identity = self.generate_task_identity()
        return self._task_identity

    def generate_task_identity(self) -> OperationIdentity:
        pass

    def check_preconditions(self):
        pass

    def on_before_delay(self):
        pass

    def check_already_running(self, reporter: ProgressStorage):
        identity = self.get_task_identity()
        if reporter.in_progress(identity):
            raise AlreadyRunning(details={'task_id': identity.id})

    def execute(self, reporter: ProgressStorage):
        try:
            reporter.report_begin_operation(self.get_task_identity())
            result = self._execute(reporter)
            reporter.report_success(self.get_task_identity(), result.dict())
            logger.info('success')
        except RestApiError as api_error:
            logger.exception(
                f'exception during async op executing: {api_error.details}, code: {api_error.error_code}, task_id: {self.get_task_identity()}'
            )
            reporter.report_failure(
                self.get_task_identity(), {'details': api_error.details, 'code': api_error.error_code}
            )
        except Exception as e:
            logger.exception(f'exception during async op executing: {e}, task_id: {self.get_task_identity()}')
            reporter.report_failure(self.get_task_identity(), {'details': str(e)})

    def _execute(self, reporter: ProgressStorage) -> T2:
        pass
