import hashlib

from django.db import transaction

from wiki.acl.check_access import assert_has_access
from wiki.acl.consts import Action
from wiki.api_v2.public.pages.exceptions import SlugOccupied, SlugReserved, IsCloudPage, ClusterBlocked
from wiki.api_v2.public.pages.page.permissions import assert_has_access_to_create_page
from wiki.api_v2.public.pages.page_identity import resolve_page_identity, PageIdentity
from wiki.api_v2.public.pages.schemas import PageSchema
from wiki.api_v2.public.pages.suggest_slug_view import is_slug_occupied
from wiki.async_operations.consts import OperationType, OperationIdentity
from wiki.async_operations.operation_executors.base import BaseOperation
from wiki.async_operations.operation_executors.clone_page.schemas import PageClone, PageCloneResponse
from wiki.async_operations.progress_storage import ProgressStorage
from wiki.pages.constants import ReservedSupertagAction
from wiki.pages.logic.clone import clone_page
from wiki.pages.models import Page
from wiki.pages.models.cluster_change import ClusterBlock, ChangeStatus
from wiki.pages.reserved_supertags import is_reserved_supertag
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.get_organization_by_inner_id import get_organization_by_inner_id
from wiki.utils.limits import assert_has_quota


class ClonePageOperation(BaseOperation[PageClone, PageCloneResponse]):
    TASK_TYPE = OperationType.CLONE_PAGE
    arg_class = PageClone
    result_class = PageCloneResponse

    def generate_task_identity(self) -> OperationIdentity:
        code = f'{self.owner.org_inner_id} | {self.owner.user_id} | {self.args.page.id} | {self.args.data.target}'
        return OperationIdentity(
            type=self.TASK_TYPE,
            id=hashlib.md5(code.encode('utf-8')).hexdigest(),
        )

    def cleanup_args(self, task_args: PageClone) -> PageClone:
        if task_args.page.slug is None or task_args.page.id is None:
            organization: BaseOrganization = get_organization_by_inner_id(self.owner.org_inner_id)
            page = resolve_page_identity(organization, page=self.args.page)

            task_args.page = PageIdentity.serialize(page)
        return task_args

    def check_preconditions(self):
        organization: BaseOrganization = get_organization_by_inner_id(self.owner.org_inner_id)
        user = organization.get_users().get(id=self.owner.user_id)
        page = resolve_page_identity(organization, page=self.args.page)
        target = self.args.data.target

        if page.page_type == Page.TYPES.CLOUD:
            raise IsCloudPage()

        if is_slug_occupied(organization, slug=target):
            raise SlugOccupied()

        if is_reserved_supertag(target, action=ReservedSupertagAction.CREATE):
            raise SlugReserved()

        assert_has_access(user, organization, page=page, op=Action.READ)
        assert_has_access_to_create_page(user, organization, slug=target)
        assert_has_quota(organization)

    def on_before_delay(self):
        self.create_cluster_block()

    def check_already_running(self, reporter: ProgressStorage):
        super().check_already_running(reporter)

        slugs = [self.args.page.slug, self.args.data.target]
        if ClusterBlock.find_blocked(slugs=slugs, org_id=self.owner.org_inner_id, in_children=False):
            raise ClusterBlocked()

    def _execute(self, reporter: ProgressStorage) -> PageCloneResponse:
        try:
            page = self.clone_page()
        finally:
            ClusterBlock.objects.filter(task_id=self.get_task_identity().id).delete()

        reporter.report_progress(self.get_task_identity(), 1, 'finished')
        return PageCloneResponse(page=PageSchema.serialize(page))

    def create_cluster_block(self):
        block = ClusterBlock(
            source=self.args.page.slug,
            target=self.args.data.target,
            status=ChangeStatus.FWD_CLONE_PHASE,
            task_id=self.get_task_identity().id,
            org_id=self.owner.org_inner_id,
            user_id=self.owner.user_id,
        )
        block.save()

    @transaction.atomic
    def clone_page(self) -> Page:
        organization = get_organization_by_inner_id(self.owner.org_inner_id)
        user = organization.get_users().get(id=self.owner.user_id)
        page = resolve_page_identity(organization, page=self.args.page)

        return clone_page(
            page=page,
            new_tag=self.args.data.target,
            authors=[user],
            last_author=user,
            organization=organization,
            copy_files=False,
            subscribe_me=self.args.data.subscribe_me,
            # as kwargs
            title=self.args.data.title or page.title,
        )
