from collections import Counter, defaultdict
from typing import DefaultDict

from wiki.async_operations.operation_executors.move_cluster.consts import MoveCluster


def compress(
    operations: list[MoveCluster],
    start: int = 0,
    states: DefaultDict[int, 'Counter[str]'] = None,
    status: dict[str, set[str]] = None,
) -> list[MoveCluster]:

    if states is None:
        states = defaultdict(Counter)
    if status is None:
        status = {'exists': set(), 'created': set()}

    optimized = operations[:start]

    for new in operations[start:]:
        last_index = len(optimized)

        # есть old.target которые можно объединить с new.source, и не перемещаем на место существующей страницы
        if states[last_index - 1][new.source] and (new.target not in status['exists'] or new.source == new.target):
            merged_index = try_merge(new, optimized, ind=last_index - 1, states=states)

            if merged_index is not None:
                optimized = compress(optimized, start=merged_index, states=states, status=status)
                continue

        optimized.append(new)
        states[last_index] = states[last_index - 1] + Counter(all_ancestors(new.source, new.target, new.next_to_slug))
        update_status_pages(source=new.source, target=new.target, status=status)

    return optimized


def try_merge(new: MoveCluster, operations: list[MoveCluster], ind: int, states: DefaultDict) -> int | None:
    old_target = new.source

    for j, op in enumerate(operations):
        if op.target == old_target:
            if same_state(old_target, states[j], states[ind]) and same_state(new.next_to_slug, states[j], states[ind]):
                op.target, op.next_to_slug, op.position = new.target, new.next_to_slug, new.position
                return j

    return None


def same_state(slug: str | None, state_one: 'Counter[str]', state_two: 'Counter[str]') -> bool:
    if slug is None:
        return True
    return state_one[slug] == state_two[slug]


def ancestors(slug: str | None) -> list[str]:
    if not slug:
        return []

    slugs = [slug[:i] for i, ch in enumerate(slug) if ch == '/']
    slugs.append(slug)
    return slugs


def all_ancestors(*slugs: str | None) -> list[str]:
    ancestors_ = []
    for slug in slugs:
        ancestors_ += ancestors(slug)
    return ancestors_


def update_status_pages(source: str, target: str, status: dict[str, set[str]]):
    created = status['created']
    exists = status['exists']

    source_ancestors = set(ancestors(source))
    if not created & source_ancestors:  # родительские страницы не были созданы
        exists |= source_ancestors

    if target not in exists:
        created.add(target)
