from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_v2.exceptions import BadRequest
from wiki.async_operations.consts import OperationOwner
from wiki.async_operations.exceptions import PreconditionsFailed
from wiki.async_operations.operation_executors.move_cluster.consts import (
    MoveClusterLegacy,
    MoveClusterRequestWithLegacy,
)
from wiki.async_operations.operation_executors.move_cluster.move_cluster import MoveClusterOperation
from wiki.async_operations.tasks.execute_async_operation import drf_view_schedule_execution
from wiki.async_operations.views import logger
from wiki.sync.connect.org_ctx import get_org
from rest_framework import serializers
from rest_framework.response import Response


class MoveClusterSerializer(serializers.Serializer):
    with_children = serializers.BooleanField(default=True)
    source = serializers.CharField(max_length=256, required=True)
    target = serializers.CharField(max_length=256, required=True)


class MoveClusterView(WikiAPIView):
    serializer_class = MoveClusterSerializer

    @raises()
    def post(self, request, *args, **kwargs):
        if self.request.user.is_anonymous:
            raise UserHasNoAccess()

        logger.info('prepare to start moving')

        serializer = MoveClusterSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        move_operation = MoveClusterLegacy(**serializer.validated_data)
        args = MoveClusterRequestWithLegacy(operations=[move_operation])
        logger.info(f'going to create async op with params: {args.dict()}')

        owner = OperationOwner(org_inner_id=get_org() and get_org().id, user_id=request.user.id)

        move_cluster_op = MoveClusterOperation(args, owner)

        try:
            task_id = drf_view_schedule_execution(move_cluster_op)
        except BadRequest as e:
            raise PreconditionsFailed.from_new_api_exception(e)

        return Response({'task_id': task_id}, 200)
