import hashlib
import logging
import os.path

from datetime import datetime
from typing import Literal, Iterable

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import Q, QuerySet

from wiki.async_operations.consts import OperationType, OperationIdentity, OperationOwner
from wiki.async_operations.operation_executors.base import BaseOperation
from wiki.async_operations.operation_executors.move_cluster.compress_operations import compress
from wiki.async_operations.operation_executors.move_cluster.consts import (
    MoveCluster,
    MoveClusterLegacy,
    MoveClusterRequestWithLegacy,
    MoveClusterResponse,
)
from wiki.async_operations.operation_executors.move_cluster.messages import (
    BECOME_NOT_ACCESSIBLE,
    CLUSTER_BLOCKED,
    CLUSTER_NOT_EXISTS,
    CLUSTER_PERMISSION_DENIED,
    DESTINATION_RESERVED,
    NEXT_TO_HAS_WRONG_CLUSTER,
    NEXT_TO_NOT_EXISTS,
    NEXT_TO_NOT_SET,
    NEXT_TO_WILL_MOVE,
    OPERATION_LIMIT_EXCEEDED,
    OVERRIDE_ATTEMPT,
    SOURCE_RESERVED,
    SLUG_IS_FIXED,
    TOO_LONG_NAME,
    bad_request_helper,
)
from wiki.async_operations.progress_storage import ProgressStorage
from wiki.pages.access import get_access_status, ACCESS_DENIED, get_bulk_raw_access
from wiki.pages.access.access_status import bulk_has_access
from wiki.pages.constants import ReservedSupertagAction as Action
from wiki.pages.logic.backlinks import rewrite_links
from wiki.pages.logic.rank import calculate_page_rank, next_page_rank
from wiki.pages.models import Page, Revision, AbsentPage, LocationHistory
from wiki.pages.models.consts import ACTUALITY_STATUS
from wiki.pages.models.cluster_change import (
    AffectedPage,
    AffectedPages,
    ChangeStatus,
    ClusterBlock,
    ClusterChange,
    Operations,
)
from wiki.pages.reserved_supertags import is_reserved_supertag
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.get_organization_by_inner_id import get_organization_by_inner_id
from wiki.utils import timezone
from wiki.utils.models import queryset_iterator
from wiki.utils.supertag import normalize_supertag

logger = logging.getLogger(__name__)
User = get_user_model()


LIMIT_OPERATIONS = 100
FIXED_SLUGS = settings.NAVIGATION_TREE_FIXED_SLUGS


class MoveClusterOperation(BaseOperation[MoveClusterRequestWithLegacy, MoveClusterResponse]):
    TASK_TYPE = OperationType.MOVE_CLUSTER
    arg_class = MoveClusterRequestWithLegacy
    result_class = MoveClusterResponse

    NUMBER_OF_STAGES = 3  # Количество этапов при переносе. Для логирования

    def cleanup_args(self, task_args: MoveClusterRequestWithLegacy) -> MoveClusterRequestWithLegacy:
        for operation in task_args.operations:
            operation.source = normalize_supertag(operation.source)
            operation.target = normalize_supertag(operation.target)
            operation.next_to_slug = normalize_supertag(operation.next_to_slug)
        return task_args

    def generate_task_identity(self) -> OperationIdentity:
        first = ','.join(f'{op.source}-{op.target}' for op in self.args.operations[:3])
        last = ','.join(f'{op.source}-{op.target}' for op in self.args.operations[-3:])

        code = f'{self.owner.org_inner_id} | {self.owner.user_id} | {len(self.args.operations)} | {first} | {last}'
        return OperationIdentity(
            type=self.TASK_TYPE,
            id=hashlib.md5(code.encode('utf-8')).hexdigest(),
        )

    def check_preconditions(self):
        changes: dict[Page, str] = {}
        user = User.objects.get(id=self.owner.user_id)

        self.assert_no_limit_operations(count=len(self.args.operations))
        for operation in self.args.operations:
            self.check_operation_preconditions(operation, changes, org_id=self.owner.org_inner_id, user=user)

    def on_before_delay(self):
        self.create_cluster_block()

    def check_already_running(self, reporter: ProgressStorage):
        super().check_already_running(reporter)
        self.assert_no_blocked_cluster(self.args.operations, org_id=self.owner.org_inner_id)

    def _execute(self, reporter: ProgressStorage) -> MoveClusterResponse:
        task_identity = self.get_task_identity()
        logger.info('[MOVER] inside _execute in move cluster operation')

        page_count = 0  # noqa

        try:
            affected_pages, compress_operations = self.move_clusters(reporter)
            page_count = len(affected_pages)
            rewrite_links(affected_pages, org_id=self.owner.org_inner_id)

            ClusterChange(
                user_id=self.owner.user_id,
                org_id=self.owner.org_inner_id,
                affected_pages=AffectedPages(pages=affected_pages).dict(),
                operations=Operations(input=self.args.operations, compress=compress_operations).dict(),
            ).save()
        finally:
            ClusterBlock.objects.filter(task_id=task_identity.id).delete()

        reporter.report_progress(self.get_task_identity(), 1, 'finished')
        return MoveClusterResponse(page_count=page_count)

    def create_cluster_block(self):
        blocks = []
        for operation in self.args.operations:
            block = ClusterBlock(
                source=operation.source,
                target=operation.target,
                status=ChangeStatus.FWD_MOVE_PHASE,
                task_id=self.get_task_identity().id,
                org_id=self.owner.org_inner_id,
                user_id=self.owner.user_id,
            )
            blocks.append(block)

        ClusterBlock.objects.bulk_create(blocks, batch_size=50)

    @classmethod
    def check_operation_preconditions(
        cls, operation: MoveCluster, changes: dict[Page, str], org_id: int | None, user: User
    ):
        source, target = operation.source, operation.target
        only_reorder = source == target

        # Дополнительные проверки
        cls.assert_have_value_next_to_slug_if_only_reorder(only_reorder, operation.next_to_slug)
        if operation.next_to_slug:
            cls.assert_not_fixed_slug(operation.next_to_slug)
            cls.assert_next_to_slug_same_cluster(operation.next_to_slug, target=target)
            cls.assert_next_to_slug_exists(operation.next_to_slug, org_id=org_id, changes=changes)
            cls.assert_next_to_slug_not_move(operation.next_to_slug, source=source)

        # Получаем страницы для перемещения
        qs = page_qs(operation, org_id=org_id, with_children=not only_reorder)
        source_pages = take_into_account_changes(source, qs=qs, changes=changes, with_children=not only_reorder)
        source_slugs = [changes.get(page, page.slug) for page in source_pages]

        # Проверяем наличие и корректность
        cls.assert_pages_and_cluster_exists(source_slugs, source)
        cls.assert_no_reserved_pages(source_slugs, kind='source')
        cls.assert_not_fixed_slug(source)
        cls.assert_source_accessible(source_pages, user, changes=changes)

        # Получаем новое расположение страниц
        if only_reorder:
            target_slugs = source_slugs
        else:
            target_slugs = get_target_slugs(source_slugs, source=source, target=target)

            cls.assert_no_reserved_pages(target_slugs, kind='target')
            cls.assert_not_fixed_slug(target)
            cls.assert_no_long_slugs(target_slugs)
            cls.assert_no_clashing_pages(target_slugs, org_id=org_id, changes=changes)

            cls.assert_target_will_be_accessible(source, target, source_pages, user, changes=changes)

        # Обновляем измененные слаги
        for page, target in zip(source_pages, target_slugs):
            changes[page] = target

    @transaction.atomic
    def move_clusters(self, reporter: ProgressStorage) -> tuple[list[AffectedPage], list[MoveCluster]]:
        now = timezone.now()

        task_identity = self.get_task_identity()
        organization: BaseOrganization = get_organization_by_inner_id(self.owner.org_inner_id)

        history: dict[int, tuple[str, str]] = {}
        redirect_pages: list[Page] = []

        operations = compress(operations=self.args.operations)
        for i, operation in enumerate(operations):
            if operation.source == operation.target:  # только изменение ранга
                rank = calculate_page_rank(operation.position, operation.next_to_slug, organization)
                organization.get_active_pages().filter(supertag=operation.source).update(rank=rank)

            else:
                qs = page_qs(operation, org_id=self.owner.org_inner_id).select_for_update()
                for page in queryset_iterator(qs, chunk_size=50):
                    old_slug: str = page.slug
                    new_slug: str = old_slug.replace(operation.source, operation.target, 1)

                    if isinstance(operation, MoveClusterLegacy):
                        redirect = self.copy_as_redirect(page, owner=self.owner, time=now)
                        redirect_pages.append(redirect)

                    # Корневая страница кластера, нужно рассчитать для неё новый rank
                    if operation.source == old_slug:
                        page.rank = calculate_rank(operation, slug=new_slug, organization=organization)

                    logger.info(f'[MOVER] [{i} / {len(operations)}] {old_slug} -> {new_slug}')
                    page.supertag = page.tag = new_slug
                    page.modified_at_for_index = now
                    page.save()

                    if page.id not in history:
                        history[page.id] = old_slug, new_slug
                    else:
                        history[page.id] = history[page.id][0], new_slug

            reporter.report_progress(task_identity, i / len(operations) / self.NUMBER_OF_STAGES)

        affected_pages = self.convert_history_to_affected_pages(history)
        self.remove_absent_pages(affected_pages, org_id=self.owner.org_inner_id)
        reporter.report_progress(task_identity, 2 / self.NUMBER_OF_STAGES)

        if not redirect_pages:
            self.create_location_history(affected_pages)
        else:  # legacy
            self.create_redirects(redirect_pages)

        return affected_pages, operations

    # meta info and redirects
    @staticmethod
    def copy_as_redirect(page, owner: OperationOwner, time: datetime) -> Page:
        page_data = dict(
            supertag=page.supertag,
            tag=page.tag,
            org_id=owner.org_inner_id,
            title=page.title,
            modified_at=time,
            modified_at_for_index=time,
            created_at=time,
            page_type=page.page_type,
            redirects_to=page,
            owner_id=owner.user_id,
            last_author_id=owner.user_id,
            mds_storage_id=page.mds_storage_id,
            formatter_version=page.formatter_version,
            actuality_status=ACTUALITY_STATUS.unspecified,
            is_documentation=page.is_documentation,
            depth=page.depth,
            rank=page.rank,
        )
        if settings.IS_INTRANET:
            page_data['is_official'] = page.is_official

        if page.page_type == Page.TYPES.CLOUD:
            page_data['page_type'] = Page.TYPES.PAGE

        return Page(**page_data)

    @staticmethod
    def convert_history_to_affected_pages(history: dict[int, tuple[str, str]]) -> list[AffectedPage]:
        affected_pages: list[AffectedPage] = []

        for page_id, (old_slug, new_slug) in history.items():
            if old_slug != new_slug:
                affected_pages.append(AffectedPage(id=page_id, slug=new_slug, previous_slug=old_slug))

        return affected_pages

    @staticmethod
    def remove_absent_pages(affected_pages: list[AffectedPage], org_id: int | None):
        logger.info('[MOVER] removing absent pages')

        created_slugs = [page.slug for page in affected_pages]
        AbsentPage.objects.filter(Q(to_supertag__in=created_slugs, from_page__org_id=org_id)).delete()

    @staticmethod
    def create_redirects(pages: list[Page]):
        logger.info('[MOVER] creating redirects')

        pages = Page.objects.bulk_create(pages, batch_size=50)

        for page in pages:
            page.authors.add(page.owner)

        Revision.objects.bulk_create([Revision.objects.produce_from_page(page) for page in pages])

    @staticmethod
    def create_location_history(affected_pages: list[AffectedPage]):
        logger.info('[MOVER] creating location histories')

        locations = [LocationHistory(page_id=page.id, slug=page.previous_slug) for page in affected_pages]
        LocationHistory.objects.bulk_create(locations, ignore_conflicts=True, batch_size=50)

    # assert-s
    @staticmethod
    def assert_no_blocked_cluster(operations: list[MoveCluster], org_id: int | None):
        sources = {operation.source for operation in operations}
        targets = {operation.target for operation in operations}
        clusters_blocked = ClusterBlock.find_blocked(slugs=list(sources | targets), org_id=org_id)

        if clusters_blocked:
            blocked = {i.source for i in clusters_blocked} | {i.target for i in clusters_blocked}
            raise bad_request_helper(CLUSTER_BLOCKED, details={'slugs': sorted(blocked)})

    @staticmethod
    def assert_no_limit_operations(count: int):
        if count > LIMIT_OPERATIONS:
            raise bad_request_helper(OPERATION_LIMIT_EXCEEDED)

    @staticmethod
    def assert_pages_and_cluster_exists(slugs: list[str], cluster_slug: str):
        if not slugs or cluster_slug not in slugs:
            raise bad_request_helper(CLUSTER_NOT_EXISTS)

    @staticmethod
    def assert_no_reserved_pages(slugs: list[str], kind: Literal['source', 'target']):
        if kind == 'source':
            err, action = SOURCE_RESERVED, Action.DELETE
        else:
            err, action = DESTINATION_RESERVED, Action.CREATE

        reserved_slugs = [slug for slug in slugs if is_reserved_supertag(slug, action)]
        if reserved_slugs:
            raise bad_request_helper(err, details={'slugs': reserved_slugs})

    @staticmethod
    def assert_no_long_slugs(slugs: list[str]):
        long_slugs = [slug for slug in slugs if len(slug) > 256]
        if long_slugs:
            raise bad_request_helper(TOO_LONG_NAME, details={'slugs': long_slugs})

    @staticmethod
    def assert_no_clashing_pages(slugs: list[str], org_id: int | None, changes: dict[Page, str]):
        changed_page_ids = {page.id for page in changes}

        clashing_pages_qs = Page.objects.filter(supertag__in=slugs, org_id=org_id).values_list('id', 'supertag')
        clashing_pages = {slug for id_, slug in clashing_pages_qs if id_ not in changed_page_ids}

        clashing_pages |= set(slugs).intersection(changes.values())

        if clashing_pages:
            raise bad_request_helper(OVERRIDE_ATTEMPT, details={'slugs': sorted(clashing_pages)})

    @staticmethod
    def assert_source_accessible(pages: list[Page], user: User, changes: dict[Page, str]):
        need_check_access = list(set(pages) - changes.keys())

        if not need_check_access:
            return

        forbidden = []
        page_cache = {}

        raw_access_list = get_bulk_raw_access(need_check_access)
        for page in need_check_access:
            if bulk_has_access(page, user, raw_access_list, privilege='read', page_cache=page_cache) == ACCESS_DENIED:
                forbidden.append(page.slug)

        if forbidden:
            raise bad_request_helper(CLUSTER_PERMISSION_DENIED, details={'slugs': forbidden})

    @staticmethod
    def assert_target_will_be_accessible(source: str, target: str, source_pages: list[Page], user: User, changes):
        # Если уже проверяли
        if source in changes.values():
            return

        # Если пользователь - это автор страницы
        source_page = next(page for page in source_pages if page.slug == source)  # всегда существует
        if user in source_page.get_authors():
            return

        # Если у пользователя есть доступ к родительской странице
        target_parent = os.path.dirname(target)

        if target_parent in changes.values():
            return

        if get_access_status(target_parent, user) != ACCESS_DENIED:
            return

        raise bad_request_helper(BECOME_NOT_ACCESSIBLE)

    @staticmethod
    def assert_next_to_slug_same_cluster(next_to_slug: str, target: str):
        if os.path.dirname(target) != os.path.dirname(next_to_slug):
            raise bad_request_helper(NEXT_TO_HAS_WRONG_CLUSTER)

    @staticmethod
    def assert_next_to_slug_exists(slug: str, org_id: int | None, changes: dict[Page, str]):
        if slug in changes.values():
            return

        page_id = Page.objects.filter(supertag=slug, org_id=org_id).values_list('id', flat=True).first()

        if page_id is None or page_id in (page.id for page in changes):  # страницы не существует, или была перемещена
            raise bad_request_helper(NEXT_TO_NOT_EXISTS)

    @staticmethod
    def assert_next_to_slug_not_move(next_to_slug: str, source: str):
        if next_to_slug.startswith(source + '/') or next_to_slug == source:
            raise bad_request_helper(NEXT_TO_WILL_MOVE)

    @staticmethod
    def assert_have_value_next_to_slug_if_only_reorder(only_reorder: bool, next_to_slug: str | None):
        if only_reorder and not next_to_slug:
            raise bad_request_helper(NEXT_TO_NOT_SET)

    @staticmethod
    def assert_not_fixed_slug(slug: str):
        if slug in FIXED_SLUGS:
            raise bad_request_helper(SLUG_IS_FIXED)


def page_qs(operation: MoveCluster | MoveClusterLegacy, org_id: int | None, with_children=True) -> QuerySet:
    if isinstance(operation, MoveClusterLegacy):
        with_children = operation.with_children

    query = Q(supertag=operation.source)
    if with_children:
        query |= Q(supertag__startswith=operation.source + '/')

    return Page.objects.filter(query, org_id=org_id, status__gt=0)


def take_into_account_changes(slug: str, qs: QuerySet, changes: dict[Page, str], with_children: bool) -> list[Page]:
    exists_qs = qs.exclude(id__in=[page.id for page in changes])
    pages = list(exists_qs)

    for page, change_slug in changes.items():
        if change_slug == slug or (with_children and change_slug.startswith(slug + '/')):
            pages.append(page)

            if not with_children:
                break

    return pages


def get_target_slugs(source_slugs: Iterable[str], source: str, target: str) -> list[str]:
    targets = []
    for slug in source_slugs:
        new_slug = slug.replace(source, target, 1)
        targets.append(new_slug)
    return targets


def calculate_rank(operation: MoveCluster, slug: str, organization: BaseOrganization) -> str:
    if operation.next_to_slug:
        page_rank = calculate_page_rank(operation.position, operation.next_to_slug, organization)
    else:
        depth = slug.count('/') + 1
        page_rank = next_page_rank(slug, depth, organization)
    return page_rank
