from typing import Dict

from django.core.cache import caches
from wiki.async_operations.consts import Progress, Status, OperationProgress, OperationIdentity, OperationOwner
from wiki.async_operations.exceptions import OperationNotFound

TTL = 60 * 60


class ProgressStorage:
    def __init__(self, key):
        self.cache = caches[key]

    def clear(self):
        self.cache.clear()

    def report_scheduled(self, identity: OperationIdentity, owner: OperationOwner):
        data = OperationProgress(
            id=identity, result={}, status=Status.SCHEDULED, owner=owner, progress=Progress(percentage=0.0, details='')
        )
        self.save(identity, data)

    def report_progress(self, identity: OperationIdentity, f: float = 0.0, s: str = 'in progress'):
        data = self.load(identity)
        data.progress = Progress(percentage=f, details=s)
        self.save(identity, data)

    def in_progress(self, identity: OperationIdentity) -> bool:
        try:
            data = self.load(identity)
        except OperationNotFound:
            return False
        return data.status in {Status.SCHEDULED, Status.IN_PROGRESS}

    def report_begin_operation(self, identity: OperationIdentity):
        data = self.load(identity)
        data.status = Status.IN_PROGRESS
        self.save(identity, data)

    def report_success(self, identity: OperationIdentity, result: Dict):
        data = self.load(identity)
        data.status = Status.SUCCESS
        data.result = result
        self.save(identity, data)

    def report_failure(self, identity: OperationIdentity, result: Dict):
        data = self.load(identity)
        if data is None:
            data = OperationProgress(
                id=identity, status=Status.FAILED, result=result, progress=Progress(percentage=0.0, details='failed')
            )
        else:
            data.status = Status.FAILED
            data.result = result
        self.save(identity, data)

    def get(self, task_id: str) -> str:
        data = self.cache.get(task_id)
        if data:
            return data
        raise OperationNotFound(OperationNotFound.debug_message + ' ' + task_id)

    def load(self, identity: OperationIdentity) -> OperationProgress:
        data = self.get(identity.id)
        return OperationProgress.parse_raw(data)

    def load_by_task_id(self, task_id: str) -> OperationProgress:
        data = self.get(task_id)
        return OperationProgress.parse_raw(data)

    def save(self, identity: OperationIdentity, data: OperationProgress):
        self.cache.set(identity.id, data.json(), TTL)


ASYNC_OP_PROGRESS_STORAGE = ProgressStorage('async_op_storage')
