import logging
from typing import Type

from ninja import Schema
from pydantic import BaseModel
from rest_framework import serializers
from rest_framework.response import Response

from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import raises
from wiki.api_v2.di import di
from wiki.api_v2.exceptions import NotFound, Forbidden
from wiki.async_operations.consts import (
    Status,
    OperationType,
    AsyncOperationStatusSchema,
    OperationCreatedSchema,
    OperationOwner,
)
from wiki.async_operations.exceptions import OperationNotFound
from wiki.async_operations.operation_executors.registry import EXECUTOR_REGISTRY
from wiki.async_operations.progress_storage import ASYNC_OP_PROGRESS_STORAGE
from wiki.async_operations.tasks.execute_async_operation import execute_async_operation
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.django_redlock.redlock import RedisLock

logger = logging.getLogger(__name__)


# /operations/move/{ABCD-EFGH-QURST}


def make_operation_status_view(type: OperationType, response_klass: Type[BaseModel]):
    @di
    def get_operation_status_view(request, organization, task_id: str) -> AsyncOperationStatusSchema:
        owner = OperationOwner(org_inner_id=organization.inner_id(), user_id=request.user.id)

        try:
            task_info = ASYNC_OP_PROGRESS_STORAGE.load_by_task_id(task_id)
        except OperationNotFound:
            raise NotFound()

        if task_info.id.type != type:
            raise NotFound('Found, but wrong type.')

        if task_info.owner is not None:  # на переходный момент
            if task_info.owner.user_id != owner.user_id or task_info.owner.org_inner_id != owner.org_inner_id:
                logger.error(f'Attempt to read task of {task_info.owner} by user {owner}')
                raise Forbidden('This task belongs to different user')

        data = AsyncOperationStatusSchema[response_klass](status=task_info.status)

        if task_info.status == Status.IN_PROGRESS:
            data.progress = task_info.progress
        elif task_info.status in {Status.SUCCESS, Status.FAILED}:
            data.result = task_info.result

        return data

    get_operation_status_view.__name__ = f'get_{type.value}_operation_status_view'

    return get_operation_status_view


def schedule_execution_view(
    request, organization: BaseOrganization, data: Schema, op_type: OperationType, dry_run: bool = False
) -> OperationCreatedSchema:
    owner = OperationOwner(org_inner_id=organization.inner_id(), user_id=request.user.id)

    executor_instance = EXECUTOR_REGISTRY[op_type](data, owner)
    logger.info(f'going to create async op with params: {data.dict()}; owner: {owner.dict()}')

    executor_instance.check_preconditions()

    with RedisLock(f'async_task {organization.inner_id()} {op_type}', blocking_timeout=1 * 30):
        executor_instance.check_already_running(ASYNC_OP_PROGRESS_STORAGE)
        task_identity = executor_instance.get_task_identity()

        if not dry_run:
            ASYNC_OP_PROGRESS_STORAGE.report_scheduled(task_identity, owner)  # race conditions otherwise
            executor_instance.on_before_delay()

            execute_async_operation.delay(
                task_type=executor_instance.TASK_TYPE,
                task_id=task_identity.id,
                args=executor_instance.args.dict(),
                owner=executor_instance.owner.dict(),
            )

    return OperationCreatedSchema(operation=task_identity, dry_run=dry_run, status_url=task_identity.build_status_url())


class AsyncRequestResultViewRecent(WikiAPIView):
    serializer_class = serializers.Serializer

    @raises()
    def get(self, request, *args, **kwargs):
        """
        Example: https://wiki-api.yandex-team.ru/_api/frontend/.async_operations?id=LEET1337
        """

        task_id = request.GET.get('id', '0')
        logger.info(f'prepare to give info about task {task_id}')
        try:
            task_info = ASYNC_OP_PROGRESS_STORAGE.load_by_task_id(task_id)
        except OperationNotFound as e:
            return Response(e.debug_message, 404)

        data = {'status': task_info.status.value}

        if task_info.status == Status.IN_PROGRESS:
            data['percentage'] = f'{task_info.progress.percentage:.1%}'
            data['details'] = task_info.progress.details
        elif task_info.status in {Status.SUCCESS, Status.FAILED}:
            data['result'] = task_info.result

        logger.info(f'ready to give info: {data}')

        return Response(data, 200)
