
import logging

from celery.result import AsyncResult

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.framework import WikiAPIView
from wiki.api_core.raises import can_raise_anything, raises
from wiki.async_process.backend.consts import AsyncTaskType
from wiki.async_process.backend.utils import wrap_async_request_result
from wiki.async_process.serializers import AsyncRequestResultViewSerializer, AsyncResponse
from wiki.async_process.tasks.async_request_processor import delay_async_request
from wiki.pages.access import is_tech_admin

logger = logging.getLogger(__name__)


def get_task_result(task_id):
    return AsyncResult(task_id)


class AsyncRequestResultView(WikiAPIView):
    serializer_class = AsyncRequestResultViewSerializer

    @can_raise_anything
    def post(self, request, *args, **kwargs):
        """
        Попытаться получить результат асинхронного запроса.

        %%
        curl -XPOST https://wiki-api.yandex-team.ru/_api/frontend/.async \
        --data {"task_id": "efa9744c-0d93-4b48-8d7f-201348393d0a"}
        %%

        Ответы могут быть следующими:
        * status: 102 - запрос еще обрабатывается
        * status: 500 - при обработке запроса произошла ошибка
        * status: 200 - в ответе возвращается JSON с результатом асинхронного запроса

        """
        data = self.validate()

        async_result = get_task_result(data['task_id'])
        if not async_result.ready():
            return AsyncResponse.not_finished()

        if async_result.status != 'SUCCESS':
            try:
                async_result.get()
            except Exception as e:
                logger.exception('Unsuccessful async request processing %s', data['task_id'])
                exc = e
            else:
                async_result.forget()
                logger.warn('No exception for unsuccessful async request %s', data['task_id'])
                exc = Exception('Unknown error')
                return super(AsyncRequestResultView, self).handle_exception(exc)

            async_result.forget()
            return super(AsyncRequestResultView, self).handle_exception(exc)

        result = async_result.get()
        response = wrap_async_request_result(result)
        async_result.forget()
        return response


class TestAsyncRequestView(WikiAPIView):
    @raises()
    def post(self, request, *args, **kwargs):
        """
        Сделать тестовый асинхронный запрос.

        Передаваемые данные могут быть любыми.

        %%
        curl -XPOST https://wiki-api.yandex-team.ru/_api/frontend/.async_test \
        --data {"key": "value"}
        %%

        Возвращает JSON с id асинхронной задачи.

        %%(js)
        {"task_id": "efa9744c-0d93-4b48-8d7f-201348393d0a"}
        %%

        """
        if not is_tech_admin(request.user):
            raise UserHasNoAccess

        return delay_async_request(task_type=AsyncTaskType.TEST, task_data=request.data)
