from typing import List, Dict, Any

from django.conf import settings
from django.core.cache import caches
from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.users import DEFAULT_AVATAR_ID
from wiki.utils.blackbox import get_users_data_by_uids
from wiki.utils.features.get_features import get_wiki_features

AVATARS_CACHE = caches['avatars_cache']


def get_avatars(user_identity: List[UserIdentity]) -> Dict[UserIdentity, str]:
    output = {}

    if get_wiki_features().avatars_enabled:
        _get_avatars(output, user_identity)

    for u in user_identity:
        if u not in output:
            output[u] = DEFAULT_AVATAR_ID

    return output


def _get_avatars(output: Dict[UserIdentity, str], user_identity: List[UserIdentity]):
    with_uid = []
    with_cloud_uid = []

    for u in user_identity:
        val = _get_from_cache(u)
        if val is not None:
            output[u] = val
            continue
        if u.uid:
            with_uid.append(u)
            continue
        if u.cloud_uid:
            with_cloud_uid.append(u)
            continue

    _populate_avatars_from_blackbox(output, with_uid)
    return output


def _get_from_cache(user_identity: UserIdentity) -> str:
    return None


def _add_to_cache(user_identity: UserIdentity, value: Any):
    pass


def _populate_avatars_from_blackbox(output: Dict, identities: List[UserIdentity]):
    if settings.IS_BUSINESS and not settings.IS_TESTS:
        lut = {identity.uid: identity for identity in identities}

        users_data = get_users_data_by_uids([identity.uid for idx, identity in identities])

        for user_data in users_data:
            if user_data.avatar.default and user_data.uid and user_data.uid in lut:
                output[lut[user_data.uid]] = user_data.avatar.default
                _add_to_cache(lut[user_data.uid], user_data.avatar.default)
