import logging
import time

from celery import Task
from celery.exceptions import SoftTimeLimitExceeded
from django.conf import settings
from pymongo import errors as pymongo_errors
from ylog.context import log_context

from wiki.utils.celery import (
    REMEMBER_CELERY_BROKER_UNAVAILABILITY_FOR,
    celery_broker_maybe_available,
    celery_broker_was_unavailable,
    set_celery_broker_available,
    set_celery_broker_unavailable,
)

HARD_AND_SOFT_LIMITS_DELTA = 5  # дельта между hard_time_limit и soft_time_limit

logger = logging.getLogger(__name__)


class CallableTask(Task):
    """
    Класс-обертка над коллабл объектом с логгированием и soft_time_limit.
    """

    abstract = True

    logger = logger  # logging.Logger
    time_limit = 600  # максимальное время выполнения таска
    soft_time_limit = None  # время выполнения таска
    # По умолчанию результат таска игнорируется!
    ignore_result = True

    def __init__(self):
        assert self.time_limit > 0  # необходимо явно определить время выполнения таска
        if self.soft_time_limit is None:
            self.soft_time_limit = soft_time_limit(self.time_limit)
        super(CallableTask, self).__init__()

    def apply_async(self, *args, **kwargs):
        """
        Запусить селери-таску только когда текущая транзакция завершится.
        """

        if settings.IS_TESTS:
            # В тестах для ускорения не будем делать проверок брокера.
            return super(CallableTask, self).apply_async(*args, **kwargs)

        if celery_broker_was_unavailable():
            if celery_broker_maybe_available():
                set_celery_broker_available()
            else:
                logger.warning('Celery broker maybe unavailable, running task synchronously')
                return super(CallableTask, self).apply(*args, **kwargs)

        try:
            return super(CallableTask, self).apply_async(*args, **kwargs)
        except pymongo_errors.PyMongoError:
            logger.exception(
                'Celery broker error, running task synchronously. '
                'Remembering unavailability for %s seconds' % REMEMBER_CELERY_BROKER_UNAVAILABILITY_FOR
            )
            set_celery_broker_unavailable()
            return super(CallableTask, self).apply(*args, **kwargs)

    def __call__(self, *args, **kwargs):
        """
        Выполнить таск.
        @rtype: anything
        @raise: пробрасывает любое исключение
        """

        with log_context(celery={'id': self.request.id, 'retries': self.request.retries, 'name': self.name}):
            try:
                r_started = time.time()

                try:
                    result = self.run(*args, **kwargs)
                except Exception as exc:
                    self.logger.exception('Celery task "%s" has unhandled exception "%s"' % (self.name, repr(exc)))
                    raise

                real = time.time() - r_started

                with log_context(celery_dur=real):
                    self.logger.info('Task "%s" has finished in %.4f secs' % (self.name, real))

                return result
            except SoftTimeLimitExceeded:
                # таск выполняется слишком долго, скоро его воркер будет убит celery.
                raise self.retry(max_retries=1, countdown=60)
            except Exception as e:
                # в случае неперехваченной ошибки есть небольшая вероятность того, что
                # запустив повторный запуск таски завершится успехом.
                if self._get_app().conf.CELERY_TASK_ALWAYS_EAGER or self._get_app().conf.CELERY_ALWAYS_EAGER:
                    raise
                raise self.retry(exc=e, max_retries=1, countdown=60)

    def run(self, *args, **kwargs):
        """
        Выполнить некую полезную нагрузку.

        функция обязательно должна принимать арги и кварги.
        Они будут переданы из аргументов таска.
        """
        raise NotImplementedError()


def soft_time_limit(hard_limit):
    """
    Вычислить стандартный soft_time_limit, не меньше 1 секунды.

    @type hard_limit: int
    @rtype: int
    """
    return max(hard_limit - HARD_AND_SOFT_LIMITS_DELTA, 1)
