import logging
import time

from django.db.models import Count, Sum

from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.models import Organization
from wiki.billing import RESTRICTION_NAMES
from wiki.org import get_org
from wiki.utils.models import get_chunked

logger = logging.getLogger(__name__)

ORG_CHUNK_SIZE = 5000
STATUS_ENABLED = Organization.ORG_STATUSES.enabled


def update_org_stats(*org_pks):
    # Заметки про возможные оптимизации.
    # 1) Сейчас статистика каждой организации обновляется
    # отдельным SQL-запросом. Несмотря на то, что статистика
    # обновляется только если она изменилась, здесь все равно
    # может быть слишком много отдельных UPDATE. Возможно, получится
    # обновлять статистику пачками.
    # 2) Можно часто обновлять статистику тех организаций, которые
    # приблизились к лимитам для своего режима, и редко обновлять для
    # тех организаций, которые еще далеки до достижения лимитов.
    # Факт достижения какого-то процента лимитов для режима организации
    # можно попробовать вычислять с помощью операций над jsonb полем.
    # 3) Кажется правильней обновлять статистику в момент создания/удаления
    # страниц, загрузки/удаления файлов, в этом случае статистика будет
    # всегда в актуальном состоянии, а таску оставить на всякий случай,
    # запускать её периодически для проверки актуальности данных

    start_time = time.time()

    orgs_query_set = Organization.objects.filter(status=STATUS_ENABLED)

    if org_pks:
        orgs_query_set = Organization.objects.filter(pk__in=org_pks)

    orgs_count = 0
    stats_updated = 0

    for org_chunk in get_chunked(orgs_query_set, ORG_CHUNK_SIZE):
        org_ids = {org.id for org in org_chunk}
        orgs_count += len(org_ids)

        if org_pks:
            org_filter = {'id__in': org_pks}
        else:
            org_filter = {'id__gte': min(org_ids), 'id__lte': max(org_ids)}

        # Извлечь за один SELECT число страниц и объем аттачей
        # не получится, т.к. при подсчете числа страниц исключаются
        # автогенеренные страницы, а при подсчете объема аттачей – нет.

        orgs_page_count = (
            Organization.objects.filter(
                status=STATUS_ENABLED,
                page__is_autogenerated=False,
                page__redirects_to__isnull=True,
                page__status=1,
                **org_filter,
            )
            .annotate(page_count=Count('page'))
            .values_list('id', 'page_count')
        )
        orgs_page_count = {org_id: org_page_count for org_id, org_page_count in orgs_page_count}

        orgs_file_size = (
            Organization.objects.filter(
                status=STATUS_ENABLED,
                page__status=1,
                page__file__status=1,
                **org_filter,
            )
            .annotate(file_size=Sum('page__file__size'))
            .values_list('id', 'file_size')
        )
        orgs_file_size = {org_id: org_file_size for org_id, org_file_size in orgs_file_size}

        for org in org_chunk:
            page_count = orgs_page_count.get(org.id, 0) or 0
            file_size = (orgs_file_size.get(org.id, 0) or 0) / (1 << 20)

            org_stats = {
                RESTRICTION_NAMES.org_page_num: page_count,
                RESTRICTION_NAMES.org_attach_size_mb: file_size,
            }

            # Обновляем запись OrgStatistics в двух случаях.
            # 1) Изменилась статистика организации, попутно пересчитываем is_limits_exceeded.
            # 2) Статистика могла не измениться, но мог измениться тарифный план организации,
            # и это может повлиять на is_limits_exceeded.
            # В остальных случаях ничего не делаем, чтобы не плодить лишних UPDATE.

            if org.orgstatistics.statistics != org_stats:
                org.orgstatistics.statistics = org_stats
                org.orgstatistics.is_limits_exceeded = _is_limits_exceeded(org)
                org.orgstatistics.save()
                stats_updated += 1
            else:
                is_limits_exceeded = _is_limits_exceeded(org)
                if org.orgstatistics.is_limits_exceeded != is_limits_exceeded:
                    org.orgstatistics.is_limits_exceeded = is_limits_exceeded
                    org.orgstatistics.save()
                    stats_updated += 1

    elapsed_time = time.time() - start_time

    logger.info('Stats updated for %d/%d orgs for %.2f seconds', stats_updated, orgs_count, elapsed_time)


class BillingLimit:
    def __init__(self, restriction):
        self.restriction = restriction

    def exceeded(self, addition, organization: BaseOrganization = None):
        limit = self.get(organization=organization)
        if limit is None or limit == -1:
            return False

        if organization:
            org = organization.as_django_model()
        else:
            org = get_org()

        if org is None:
            return False

        value = org.orgstatistics.statistics[self.restriction] + addition
        return value > limit

    def get(self, organization: BaseOrganization = None):
        if organization:
            org = organization.as_django_model()
        else:
            org = get_org()

        if org is None:
            return -1

        return org.mode.limits[self.restriction]


def _is_limits_exceeded(org):
    for restriction_name in org.orgstatistics.statistics:
        stat = org.orgstatistics.statistics[restriction_name]
        limit = org.mode.limits[restriction_name]
        if (limit is not None and limit != -1) and stat > limit:
            return True
    return False
