import logging

import waffle
from django.conf import settings
from wiki.api_core.waffle_switches import ELASTIC
from wiki.cloudsearch.exceptions import CloudsearchIsDownError, CloudsearchBadRequest
from wiki.cloudsearch.indexable_model_mixin import IndexableModelMixin
from wiki.cloudsearch.sqs_client import SqsClient
from wiki.cloudsearch.tvm_client import TvmServiceClient
from wiki.utils.errors import ApiRequestFailed, ApiRequestBadRequest

logger = logging.getLogger(__name__)


def only_switch(switch_name):
    def _only_switch(f):
        def inner(*args, **kwargs):
            if waffle.switch_is_active(switch_name):
                return f(*args, **kwargs)

        return inner

    return _only_switch


class CloudSearchClient(TvmServiceClient, SqsClient):
    TVM_CLIENT_ID = settings.CLOUDSEARCH_TVM_CLIENT_ID
    HOST = settings.CLOUDSEARCH_HOST
    SQS_QUEUE_NAME = settings.SQS_QUEUE_NAME
    REMOTE_NAME = 'Cloudsearch'
    batch = []

    @only_switch(ELASTIC)
    def index_batch(self, indexable_model: IndexableModelMixin):
        if len(self.batch) == settings.CLOUDSEARCH_BATCH_SIZE:
            self.send_batch()
        self.batch.append({'uuid': indexable_model.get_search_uuid(), 'type': 'upsert'})

    @only_switch(ELASTIC)
    def send_batch(self):
        if len(self.batch):
            self._send_message(data={'batch': self.batch, 'type': 'index_batch'})
            self.batch.clear()

    @only_switch(ELASTIC)
    def on_model_upsert(self, indexable_model: IndexableModelMixin):
        self._send_message(data={'uuid': indexable_model.get_search_uuid(), 'type': 'upsert'})

    @only_switch(ELASTIC)
    def on_model_delete(self, indexable_model: IndexableModelMixin):
        self._send_message(data={'uuid': indexable_model.get_search_uuid(), 'type': 'delete'})
        self._apply_to_children_files(indexable_model, 'delete')

    @only_switch(ELASTIC)
    def on_model_update_acl(self, indexable_model: IndexableModelMixin):
        self._send_message(data={'uuid': indexable_model.get_search_uuid(), 'type': 'update_acl'})
        self._apply_to_children_files(indexable_model, 'update_acl')

        # Если это страница - надо рекурсивно обновить acl у всех дочерних страниц и всех связанных файлов
        self._apply_to_children_pages(indexable_model, 'update_acl')

    def _apply_to_children_pages(self, indexable_model: IndexableModelMixin, action: str):
        for child_page in indexable_model.get_children_pages():
            self._send_message(data={'uuid': child_page.get_search_uuid(), 'type': action})
            self._apply_to_children_files(child_page, action)

    def _apply_to_children_files(self, indexable_model: IndexableModelMixin, action: str):
        for file in indexable_model.get_children_files():
            self._send_message(data={'uuid': file.get_search_uuid(), 'type': action})

    @classmethod
    def query_view(cls, json_):
        try:
            return cls._call_api(endpoint='/api/query', json=json_)
        except ApiRequestBadRequest:
            raise CloudsearchBadRequest()
        except ApiRequestFailed:
            raise CloudsearchIsDownError()


CLOUD_SEARCH_CLIENT = CloudSearchClient()
