import json
import logging

from django.conf import settings
import boto3
from typing import Callable

logger = logging.getLogger(__name__)


# Декоратор вызывает функцию под try
def safe_invoke(function: Callable) -> Callable:
    def inner(client, data):
        try:
            return function(client, data)
        except Exception as e:
            logger.exception(f'Exception during safe_invoke: {e}')

    return inner


class SqsClient:
    _sqs_client = None
    _sqs_queue_url = None
    SQS_QUEUE_NAME: str

    def get_sqs_client(self):
        if self._sqs_client is None:
            self._sqs_client = boto3.client(
                'sqs',
                region_name='eu-west-1',
                endpoint_url=settings.SQS_ENDPOINT_URL,
                aws_access_key_id=settings.SQS_AWS_ACCESS_KEY,
                aws_secret_access_key=settings.SQS_AWS_SECRET_ACCESS_KEY,
                aws_session_token=settings.SQS_AWS_SESSION_TOKEN,
            )
            self._sqs_queue_url = self._sqs_client.get_queue_url(QueueName=self.SQS_QUEUE_NAME)['QueueUrl']

        return self._sqs_client

    @safe_invoke
    def _send_message(self, data):
        logger.info(f'sending to cloudsearch queue: {data}')
        return self.get_sqs_client().send_message(QueueUrl=self._sqs_queue_url, MessageBody=json.dumps(data))
