import datetime
import logging
from functools import cache
from typing import Dict

from django.db import connection
from wiki.cloudsearch.cloudsearch_client import CLOUD_SEARCH_CLIENT
from wiki.cloudsearch.indexable_model_mixin import IndexableModelMixin
from wiki.files.models import File
from wiki.grids.models import Grid
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.models.page import Page
from wiki.sync.connect.models import Organization
from wiki.cloudsearch.active_orgs import ORGS

logger = logging.getLogger(__name__)
MODELS_MAPPER = {'wp:': Page, 'wf:': File, 'wg:': Grid}


def find_by_search_uuid(search_uuid: str) -> IndexableModelMixin:
    index = int(search_uuid[3:])
    prefix = search_uuid[:3]
    return MODELS_MAPPER[prefix].objects.get(id=index)


def render_document_for_indexation(obj: IndexableModelMixin) -> Dict:
    return {
        'uuid': obj.get_search_uuid(),
        'document': obj.get_document(),
        'metadata': obj.get_metadata(),
        'acl': calculate_acl(obj.get_acl_subject()),
    }


def calculate_acl(acl_subject):
    raw_access = get_bulk_raw_access([acl_subject])
    interpreted_access = interpret_raw_access(raw_access[acl_subject])

    is_restricted = interpreted_access['is_restricted'] or interpreted_access['is_owner']
    group_ids = [group.id for group in interpreted_access['groups']]

    if is_restricted:
        uids = [a.get_uid() for a in acl_subject.get_authors() if a.get_uid() is not None]
        cloud_uids = [a.get_cloud_uid() for a in acl_subject.get_authors() if a.get_cloud_uid() is not None]
    else:
        uids = []
        cloud_uids = []

    if interpreted_access['is_restricted']:
        uids.extend([el.user.get_uid() for el in interpreted_access['users'] if el.user.get_uid() is not None])
        cloud_uids.extend(
            [el.user.get_cloud_uid() for el in interpreted_access['users'] if el.user.get_cloud_uid() is not None]
        )

    uids = list(set(uids))
    cloud_uids = list(set(cloud_uids))

    return {
        'is_restricted': is_restricted,
        'org_id': acl_subject.org.dir_id,
        'uids': uids,
        'cloud_uids': cloud_uids,
        'group_ids': group_ids,
    }


def reindex_all_active_orgs(days: int, force=False):
    """
    Applies reindex_all_org to all orgs that have last_usage in ;

    interval: 'X MONTH|YEAR', for example: 3 MONTH
    """

    if days < 1 or not isinstance(days, int):
        logger.warning(f'wrong days count: {days}, natural number estimated')
        return

    for org in Organization.objects.filter(last_usage__gt=datetime.datetime.utcnow() - datetime.timedelta(days=days)):
        logger.info(f'now indexing org with dir_id: {org.dir_id}')
        reindex_all_org(org.dir_id, force)


def reindex_all_active_orgs_via_sql(force=False):
    for dir_id in ORGS:
        org_id = Organization.objects.get(dir_id=dir_id).id
        reindex_all_org_via_sql(org_id, force)


# noinspection SqlDialectInspection
def reindex_all_org_via_sql(org_id, force=False):
    org = Organization.objects.get(id=org_id)
    if org.is_indexed and not force:
        return

    with connection.cursor() as cursor:
        sql_query = """SELECT id
                        from (
                                 SELECT pages_page.id, 2 as rev_count
                                 FROM pages_page
                                 WHERE pages_page.org_id = %s
                                   AND pages_page.supertag != 'homepage'
                                   AND pages_page.supertag != 'clusterusers'
                                   AND pages_page.supertag != 'users'
                                   AND pages_page.supertag != 'sandbox'
                                   AND NOT (pages_page.supertag LIKE 'users/%%')
                                   AND NOT (pages_page.supertag LIKE 'sandbox/%%')
                                 UNION
                                 SELECT pages_page.id, count(1) as rev_count
                                 FROM pages_page
                                          JOIN pages_revision ON pages_revision.page_id = pages_page.id
                                 WHERE pages_page.org_id = %s
                                   AND (pages_page.supertag = 'homepage' OR pages_page.supertag LIKE 'users/%%')
                                 GROUP BY pages_page.id
                             ) as subq
                        WHERE subq.rev_count > 1;"""
        params = [org_id] * 2
        cursor.execute(sql_query, params)
        rows = cursor.fetchall()
        indexable_types = {Page.TYPES.PAGE, Page.TYPES.GRID}
        for row in rows:
            page_id = row[0]
            page = Page.objects.get(id=page_id)
            if page.page_type not in indexable_types:
                continue

            CLOUD_SEARCH_CLIENT.index_batch(page)
            for file in page.file_set.all():
                CLOUD_SEARCH_CLIENT.index_batch(file)

        CLOUD_SEARCH_CLIENT.send_batch()  # flush

    org.is_indexed = True
    org.save()


def reindex_all_org(dir_id, force=False):
    org = Organization.objects.get(dir_id=dir_id)
    if org.is_indexed and not force:
        return

    for page in org.page_set.all():
        if page.page_type == Page.TYPES.PAGE and not is_ignored(page) and not is_etalon(page, get_etalon_pages()):
            CLOUD_SEARCH_CLIENT.on_model_upsert(page)
            for file in page.file_set.all():
                CLOUD_SEARCH_CLIENT.on_model_upsert(file)

    for grid in Grid.objects.filter(org=org):
        CLOUD_SEARCH_CLIENT.on_model_upsert(grid)

    org.is_indexed = True
    org.save()


def is_ignored(page: Page):
    return page.supertag.startswith('sandbox')


def is_etalon(page: Page, etalon_pages):
    return page.body in etalon_pages


@cache
def get_etalon_pages():
    logger.info('cloudsearch: getting etalon pages')
    etalon_pages = set()
    for el in Page.objects.filter(supertag__startswith='etalon/ru/personal'):
        etalon_pages.add(el.body)
    for el in Page.objects.filter(supertag__startswith='etalon/en/personal'):
        etalon_pages.add(el.body)
    for el in Page.objects.filter(supertag__startswith='etalon/homepage'):
        etalon_pages.add(el.body)
    return etalon_pages


# noinspection SqlDialectInspection
def update_last_usages():
    with connection.cursor() as cursor:
        cursor.execute(
            """
            WITH cte AS (
                  SELECT
                    pages_page.org_id,
                    MAX(pages_revision.created_at) AS calculated
                  FROM
                    pages_revision
                    INNER JOIN pages_page ON pages_page.id = pages_revision.page_id
                    INNER JOIN dir_data_sync_organization ON pages_page.org_id = dir_data_sync_organization.id
                    AND dir_data_sync_organization.last_usage is NULL
                  GROUP BY
                    pages_page.org_id
                  LIMIT
                    1000
                )
            UPDATE
              dir_data_sync_organization
            SET
              last_usage = cte.calculated
            FROM cte
                WHERE
                  dir_data_sync_organization.id = cte.org_id;"""
        )
        logger.info(f'set {cursor.rowcount} organizations')
